// ----------------------------------------------------------------------------
// FILE NAME: itrace.hpp
//
// DESCRIPTION:
//   Declaration of the classes:
//     ITrace
//
// COPYRIGHT:
//   IBM Open Class Library
//   Licensed Materials - Property of IBM
//
//   5645-001
//   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.
//
// Revision: 06  1.6.1.4  source/core/base/itrace.hpp, appsupport, ioc.v400, 980918  
// ----------------------------------------------------------------------------

#ifndef _ITRACE_
#define _ITRACE_

#include <ibase.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma enum(4)
#pragma pack(push,4)

class IString;

class IC_EXPORTB ITrace
{
friend class ITraceSetup;

public:
/*------------------------- Constructors -------------------------------------*/
  ITrace                   ( const char* traceName =0,
                             long        lineNumber=0 );

 ~ITrace                   ( );

/*-------------------------- Output/Location ---------------------------------*/
static void
  write                    ( const IString& text ),
  write                    ( const char*    text );

enum Destination           {
  queue,
  standardError,
  standardOutput,
  file
  };

static ITrace::Destination
  traceDestination         ( );

static void
  writeToQueue             ( ),
  writeToStandardError     ( ),
  writeToStandardOutput    ( ),
  writeToFile              ( );

/*------------------------- Enable/Disable -----------------------------------*/
static void
  enableTrace              ( ),
  disableTrace             ( );

static bool
  isTraceEnabled           ( );

/*------------------------- Formatting ---------------------------------------*/
static void
  enableWriteLineNumber    ( ),
  disableWriteLineNumber   ( ),
  enableWritePrefix        ( ),
  disableWritePrefix       ( );

static bool
  isWriteLineNumberEnabled ( ),
  isWritePrefixEnabled     ( );

/*------------------------- Debugging ----------------------------------------*/
static void
  writeDebugLocation       (const char* str, const IExceptionLocation& location);

protected:
/*------------------------- Thread ID ----------------------------------------*/
static unsigned long
  threadId                 ( );

/*------------------------- Protected Output Operations ----------------------*/
static void
  writeString              ( char*          text   ),
  writeFormattedString     ( const IString& string,
                             char*          marker );

private:
/*------------------------- Private ------------------------------------------*/

enum State {
  uninitialized=1,
  on=2,
  writeLineNumber=4,
  writePrefix=8
  };
char
 *pszClTraceName;
static int
  iClState;
static ITrace::Destination
  iClTraceLocation;
static unsigned long
  remainingStack ( );

};

#pragma pack(pop)
#pragma enum(pop)

#define IC_TRACE_RUNTIME

#ifdef IC_DEVELOP
  #define IC_TRACE_DEVELOP
#endif

#ifdef IC_TRACE_ALL
  #define IC_TRACE_DEVELOP
#endif

#ifndef __FUNCTION__
  #define __FUNCTION__ "f"
#endif

#ifdef IC_TRACE_ALL
   #define IMODTRACE_ALL(modname)  ITrace trc(modname, __LINE__ )
   #define IFUNCTRACE_ALL()        ITrace trc(__FUNCTION__, __LINE__ )
   #define ITRACE_ALL(p1)          ITrace::write(p1)
   #ifndef IC_TRACE_DEVELOP
     #define IC_TRACE_DEVELOP
   #endif
#else
   #define IMODTRACE_ALL(modname)
   #define IFUNCTRACE_ALL()
   #define ITRACE_ALL(p1)
#endif

#ifdef IC_TRACE_DEVELOP
   #define IMODTRACE_DEVELOP(modname) ITrace trc(modname,  __LINE__ )
   #define IFUNCTRACE_DEVELOP()       ITrace trc(__FUNCTION__, __LINE__ )
   #define ITRACE_DEVELOP(p1)         ITrace::write(p1)
   #ifndef IC_TRACE_RUNTIME
     #define IC_TRACE_RUNTIME
   #endif
#else
   #define IMODTRACE_DEVELOP(modname)
   #define IFUNCTRACE_DEVELOP()
   #define ITRACE_DEVELOP(p1)
#endif

#ifdef IC_TRACE_RUNTIME
   #define IMODTRACE_RUNTIME(modname)  ITrace trc(modname, __LINE__ )
   #define IFUNCTRACE_RUNTIME()        ITrace trc(__FUNCTION__, __LINE__ )
   #define ITRACE_RUNTIME(p1)          ITrace::write(p1)
#else
   #define IMODTRACE_RUNTIME(modname)
   #define IFUNCTRACE_RUNTIME()
   #define ITRACE_RUNTIME(p1)
#endif

#ifdef IC_TRACE_DEVELOP
   #define ITRACE_MOTIF_NOP() \
               ITrace::writeDebugLocation("Motif NOP", IEXCEPTION_LOCATION() )
   #define ITRACE_MOTIF_NOTYET() \
               ITrace::writeDebugLocation("Motif NOTYET", IEXCEPTION_LOCATION() )
   #define ITRACE_WIN_NOP()      \
               ITrace::writeDebugLocation("Windows NOP", IEXCEPTION_LOCATION() )
   #define ITRACE_WIN_NOTYET()   \
               ITrace::writeDebugLocation("Windows NOTYET", IEXCEPTION_LOCATION() )
   #define ITRACE_PM_NOP()       \
               ITrace::writeDebugLocation("OS/2 NOP", IEXCEPTION_LOCATION() )
   #define ITRACE_PM_NOTYET()    \
               ITrace::writeDebugLocation("OS/2 NOTYET", IEXCEPTION_LOCATION() )
#else
   #define ITRACE_MOTIF_NOP()
   #define ITRACE_MOTIF_NOTYET()
   #define ITRACE_WIN_NOP()
   #define ITRACE_WIN_NOTYET()
   #define ITRACE_PM_NOP()
   #define ITRACE_PM_NOTYET()
#endif

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#include <itrace.inl>

#endif
