// Revision: 62 1.36.1.8 source/core/text/itext/istylset.hpp, text, ioc.v400, 980918 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//      IStylSet.hpp    (IStyleSet.h)
//
//  Contains the class that represents an aggregate of styles as applied to a given range of text.
//      (ITextStyleRunIterator is currently also defined in here.)
//      ITextStyleSet                           - Represents an aggregate of styles applied to a given range of text
//              ITextStyleRunIterator           - Allows the user to iterate through the style runs in an IText
//
//      IBM Open Class Library
//      (C) Copyright International Business Machines Corporation 1997
//      Licensed Material - Program-Property of IBM - All Rights Reserved.
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#ifndef _ISTYLSET_
#define _ISTYLSET_

#include "istyles.hpp"
#include "itext.hpp"
#include "imrefcnt.hpp"
#include "itxtplmt.hpp"
#include "imtabrl.hpp"

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

template <class AnUnderlyingType> class ICrossOverNodeOf;

class ICharacterRenderer;
class IParagraphRenderer;

#pragma enum(4)
#pragma pack(push,4)

//========================================================================================
// CLASS ITextStyleSet
//      A class representing a collection of text styles.
//========================================================================================
class ITextStyleSet: public IMRefCounted, public IMSTREAMABLE {
                StreamableDeclarationsMacro(ITextStyleSet);

        public:
                                                                ITextStyleSet();
                                                                ITextStyleSet(const ITextStyle& style);
                                                                ITextStyleSet(const ITextStyleSet&      other);

                                                                ~ITextStyleSet();

                ITextStyleSet&                  operator=(const ITextStyleSet&  other);
                ITextStyleSet&                  operator=(const ITextStyle&     style);

                void                                    operator>>=(IDataStream&        toWhere) const;
                void                                    operator<<=(IDataStream&        fromWhere);

                bool                                    operator==(const ITextStyleSet& other) const;
                bool                                    operator!=(const ITextStyleSet& other) const;

                ITextStyleSet&                  operator+=(const ITextStyleSet& other);
                ITextStyleSet&                  operator+=(const ITextStyle&    style);

                void                                    add(const ITextStyleSet&        other);
                void                                    add(const ITextStyle&   style);

                ITextStyleSet                   operator+(const ITextStyleSet&  other) const;
                ITextStyleSet                   operator+(const ITextStyle&     style) const;

                ITextStyleSet&                  operator-=(const ITextStyleSet& other);
                ITextStyleSet&                  operator-=(const ITextStyle&    style);

                void                                    remove(const ITextStyleSet&     other);
                void                                    remove(const ITextStyle&        style);
                void                                    removeByValue(const ITextStyleSet&      other);
                void                                    removeByValue(const ITextStyle& style);
                void                                    removeAll();

                ITextStyleSet                   operator-(const ITextStyleSet&  other) const;
                ITextStyleSet                   operator-(const ITextStyle&     style) const;

                bool                                    empty() const;

                size_t                                  styleCount() const;

                bool                                    extract(ITextStyle&     style) const;
                bool                                    extract(        ITextStyleSet&                                  other,
                                                                                        ITextStyle::EStylePropagation   propagation) const;

                bool                                    contains(const ITextStyle&      style) const;
                bool                                    contains(const ITextStyleSet&   other) const;
                bool                                    contains(ITextStyle::EStylePropagation  propagation) const;
                bool                                    containsByValue(const ITextStyle&       style) const;
                bool                                    containsByValue(const ITextStyleSet&    other) const;

                void                                    intersect(const ITextStyleSet&  other);
                void                                    intersectByValue(const ITextStyleSet&   other);

                void                                    filterByType(ITextStyle::EStylePropagation      typeToKeep);

                void                                    printDebugInfo() const;

                static ITextStyleSet&   emptyStyleSet();

        private:
                virtual void                    writeToStream(IDataStream&      toWhere) const;
                virtual void                    readFromStream(IDataStream&     fromWhere);

                void*                                   copyKludge() const;

                void                                    setContainmentFlag(unsigned long        mask);
                void                                    clearContainmentFlag(unsigned long      mask);
                bool                                    testContainmentFlag(unsigned long       mask) const;

                void                                    setFloatValue(  short   index,
                                                                                                float   newValue);
                float                                   floatValue(     short   index) const;

                void                                    setBooleanValue(        unsigned long   mask,
                                                                                                        bool                    newValue);
                bool                                    booleanValue(   unsigned long   mask) const;

                void                                    setColorComponent(      short                   index,
                                                                                                        unsigned char   newValue);
                unsigned char                   colorComponent( short   index) const;

                void                                    setTypefaceValue(const IText&   newValue);
                IText                                   typefaceValue() const;

                void                                    setJustificationValue(IParagraphJustificationStyle::EJustification      newValue);
                IParagraphJustificationStyle::EJustification
                                                                justificationValue() const;

                void                                    setSpreadValue(IParagraphSpreadStyle::ESpread   newValue);
                IParagraphSpreadStyle::ESpread
                                                                spreadValue() const;

                void                                    setCharacterRenderer(const ICharacterRenderer& newValue);
                ICharacterRenderer*     characterRenderer() const;

                void                                    setParagraphRenderer(const IParagraphRenderer& newValue);
                IParagraphRenderer*     paragraphRenderer() const;

                enum {          // style-containment mask values
                        // Boolean styles
                        kHasBoldfaceStyle                       = 0x00000001,
                        kHasItalicStyle                         = 0x00000002,
                        kHasOutlineStyle                        = 0x00000004,
                        kHasUnderlineStyle                      = 0x00000008,
                        kHasStrikethroughStyle         		= 0x00000010,
                        kHasUneditableStyle                     = 0x00000020,
                        kHasBitmapStyle                         = 0x00000040,
                        kHasFixedPitchStyle                     = 0x00000080,

                        // float styles
                        kHasPointSizeStyle                      = 0x00000100,
                        kHasSuperSubscriptStyle                 = 0x00000200,
                        kHasStartIndentStyle                    = 0x00000400,
                        kHasEndIndentStyle                      = 0x00000800,
                        kHasFirstIndentStyle                    = 0x00001000,
                        kLowestFloatStyleFlag                   = kHasPointSizeStyle,
                        kHighestFloatStyleFlag                  = kHasFirstIndentStyle,

                        // miscellaneous styles
                        kHasTypefaceStyle                       = 0x00010000,
                        kHasColorStyle                          = 0x00020000,
                        kHasJustificationStyle                  = 0x00040000,
                        kHasSpreadStyle                         = 0x00080000,
                        kHasCharacterRenderer                   = 0x00100000,
                        kHasParagraphRenderer                   = 0x00200000,
                        kHasTabRuler                            = 0x00400000,

                        kAllBooleanStyles                       = 0x000000ff,
                        kAllFloatStyles                         = 0x00001f00,
                        kAllCharacterStyles                     = 0x001303ff,
                        kAllParagraphStyles                     = 0x006c1c00,
                        kAllStyles                              = 0x007f1fff,
                        kNoStyles                               = 0
                };

                enum {          // boolean-style mask values (for the moment, I assume that all Boolean style values
                                        // occupy the same bits in fBooleanStyleValues that their containment flags to in
                                        // fStyleContainmentFlags.  This may have to change as we add more styles.)
                        kBoldface                                       = 1,
                        kItalic                                         = 2,
                        kOutline                                        = 4,
                        kUnderline                                      = 8,
                        kStrikethrough                          	= 16,
                        kUneditable                                     = 32,
                        kBitmap                                         = 64,
                        kFixedPitch                                     = 128
                };

                enum {          // array indices for "float" values
                        kPointSize                                      = 0,
                        kSuperSubscript,
                        kStartIndent,
                        kEndIndent,
                        kFirstIndent,
                        kLowestFloatStyleIndex          = kPointSize,
                        kHighestFloatStyleIndex         = kFirstIndent
                };

                enum {          // array indices for color components
                        kRed                                            = 0,
                        kGreen,
                        kBlue
                };

                unsigned long                   fStyleContainmentFlags;
                unsigned long                   fBooleanStyleValues;
                float                                   fFloatStyleValues[5];
                unsigned char                   fColorStyleComponents[3];

                IText                                   fTextTypefaceStyleValue;
                IParagraphJustificationStyle::EJustification
                                                                fParagraphJustificationStyleValue;
                IParagraphSpreadStyle::ESpread
                                                                fParagraphSpreadStyleValue;

                ICharacterRenderer*             fCharacterRenderer;
                IParagraphRenderer*             fParagraphRenderer;
                IMTabRuler*                             fRuler;

                static  ITextStyleSet*  fgEmptyStyleSet;

                friend class ITextStyleSetIterator;
                friend class ICrossOverNodeOf<ITextStyleSet>;

                friend class ITextTypefaceStyle;
                friend class ITextPointSizeStyle;
                friend class ITextBoldfaceStyle;
                friend class ITextItalicStyle;
                friend class ITextUnderlineStyle;
                friend class ITextStrikethroughStyle;
                friend class ITextOutlineStyle;
                friend class ITextUneditableStyle;
                friend class ITextBitmapStyle;
                friend class ITextColorStyle;
                friend class ITextSuperSubscriptStyle;
                friend class ITextFixedPitchStyle;
                friend class IParagraphJustificationStyle;
                friend class IParagraphSpreadStyle;
                friend class IParagraphStartIndentStyle;
                friend class IParagraphEndIndentStyle;
                friend class IParagraphFirstIndentStyle;

                friend class ICharacterRenderer;
                friend class IParagraphRenderer;
                friend class ITabRulerStyle;
};

#include <istylset.inl>


//========================================================================================
// CLASS ITextStyleRunIterator
//      A convenience class for walking through the style runs in an IText.
//========================================================================================
class ITextStyleRunIterator {
        public:
                typedef size_t length_type;
                typedef size_t offset_type;

                                                                                        ITextStyleRunIterator(  const IText&    theIText,
                                                                                                                                        ITextStyle::EStylePropagation
                                                                                                                                                                        propagation = ITextStyle::kMaximumPropagation);
                                                                                        ITextStyleRunIterator(const ITextStyleRunIterator&      that);
                                                                                        ~ITextStyleRunIterator();

                bool                                                            operator==(const ITextStyleRunIterator& that) const;
                bool                                                            operator!=(const ITextStyleRunIterator& that) const;

                const ITextStyleSet&                            operator*() const;
                const ITextStyleSet*                            operator->() const;

                length_type                                                     runLength() const;
                offset_type                                                     runStart() const;

                                                                                        operator bool() const;
                static const ITextStyleRunIterator      end();

                ITextStyleRunIterator&                          operator++();
                ITextStyleRunIterator                           operator++(int);
                ITextStyleRunIterator&                          operator--();
                ITextStyleRunIterator                           operator--(int);

                void                                                            setCharacterOffset(offset_type  charOffset);
                void                                                            setText(const IText&    theText);
                void                                                            setPropagation(ITextStyle::EStylePropagation    propagation);

        private:
                                                                                        ITextStyleRunIterator();
                ITextStyleRunIterator&                          operator=(const ITextStyleRunIterator&);

                const IText*                                            fText;
                ITextStyle::EStylePropagation           fPropagation;
                offset_type                                                     fRunOffset;
                length_type                                                     fRunLength;
                ITextStyleSet                                           fRunStyles;
                bool                                                            fIsValid;
};

#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _ISTYLSET_
