// Revision: 58 1.22.1.5 source/core/text/itext/istyles.hpp, text, ioc.v400, 980918 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//      IStyles.hpp
//
//  Contains the various concrete text style classes currently supported by IText.
//              ITextTypefaceStyle                              - Specifies the typeface of a range of text
//              ITextPointSizeStyle                             - Specifies the point size of a range of text
//              ITextBoldfaceStyle                              - Specifies whether a range of text is in boldface
//              ITextItalicStyle                                - Specifies whether a range of text is in italics
//              ITextUnderlineStyle                             - Specifies whether a range of text is underlined
//              ITextStrikethroughStyle                 - Specifies whether a range of text has a strikethrough
//              ITextOutlineStyle                               - Specifies whether a range of text is outlined
//              ITextUneditableStyle                    - Specifies whether a range of text is uneditable
//              ITextBitmapStyle                                - Specifies whether to prefer bitmap fonts over outline fonts
//              ITextColorStyle                                 - Specifies the color of a range of text
//              ITextSuperSubscriptStyle                - Specifies the amount of baseline shift of a range of text
//              ITextFixedPitchStyle                            - Specifies the fixed pitch font
//              IParagraphJustificationStyle    - Specifies the justification of a range of paragraphs
//              IParagraphSpreadStyle                   - Specifies the spread of a range of paragraphs
//              IParagraphStartIndentStyle              - Specifies the starting indent distance of a range of paragraphs
//              IParagraphEndIndentStyle                - Specifies the ending indent distance of a range of paragraphs
//              IParagraphFirstIndentStyle              - Specifies the first-line indent of a range of paragraphs
//              ITabRulerStyle                                  - Specifies tab stops of a range of paragraphs
//
//              IBM Open Class Library
//              (C) Copyright International Business Machines Corporation 1997
//              Licensed Material - Program-Property of IBM - All Rights Reserved.
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#ifndef _ISTYLES_
#define _ISTYLES_

#include <itext.hpp>
#include <imtabrl.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

class ITextStyleSet;

#pragma enum(4)
#pragma pack(push,4)

//========================================================================================
// CHARACTER STYLES
//========================================================================================

//========================================================================================
// CLASS ITextTypefaceStyle
//      A style that controls the typeface of a range of text.
//========================================================================================
class ITextTypefaceStyle : public ICharacterStyle {
        public:
                                                                                ITextTypefaceStyle();
                                                                                ITextTypefaceStyle(const IText& name);
                                                                                ITextTypefaceStyle(const ITextTypefaceStyle& other);

                virtual                                                 ~ITextTypefaceStyle();

                ITextTypefaceStyle&                             operator=(const ITextTypefaceStyle&     other);

                bool                                                    operator==(const ITextTypefaceStyle&    other) const;
                bool                                                    operator!=(const ITextTypefaceStyle&    other) const;

                IText                                                   typeface() const;
                void                                                    setTypeface(const IText&        value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                IText                                                   fValue;

};

//========================================================================================
// CLASS ITextPointSizeStyle
//      A style that controls the point size of a range of text.
//========================================================================================
class ITextPointSizeStyle : public ICharacterStyle {
        public:
                                                                                ITextPointSizeStyle();
                                                                                ITextPointSizeStyle(float       value);
                                                                                ITextPointSizeStyle(const ITextPointSizeStyle&  other);

                virtual                                                 ~ITextPointSizeStyle();

                ITextPointSizeStyle&                    operator=(const ITextPointSizeStyle&    other);

                bool                                                    operator==(const ITextPointSizeStyle&   other) const;
                bool                                                    operator!=(const ITextPointSizeStyle&   other) const;

                float                                                   pointSize() const;
                void                                                    setPointSize(float      value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                float                                                   fValue;
};

//========================================================================================
// CLASS ITextBoldfaceStyle
//  A style that determines whether the text it is applied to is boldface or lightface (true means bold).
//========================================================================================
class ITextBoldfaceStyle : public ICharacterStyle {
        public:
                                                                                ITextBoldfaceStyle();
                                                                                ITextBoldfaceStyle(bool isSet);
                                                                                ITextBoldfaceStyle(const ITextBoldfaceStyle&    other);

                virtual                                                 ~ITextBoldfaceStyle();

                ITextBoldfaceStyle&                             operator=(const ITextBoldfaceStyle&     other);

                bool                                                    operator==(const ITextBoldfaceStyle&    other) const;
                bool                                                    operator!=(const ITextBoldfaceStyle&    other) const;

                bool                                                    isBoldface() const;
                void                                                    setBoldface(bool        value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextItalicStyle
//      A style that determines whether the text it is applied to is italic or roman (true means italic).
//========================================================================================
class ITextItalicStyle : public ICharacterStyle {
        public:
                                                                                ITextItalicStyle();
                                                                                ITextItalicStyle(bool   isSet);
                                                                                ITextItalicStyle(const ITextItalicStyle&        other);

                virtual                                                 ~ITextItalicStyle();

                ITextItalicStyle&                               operator=(const ITextItalicStyle&       other);

                bool                                                    operator==(const ITextItalicStyle&      other) const;
                bool                                                    operator!=(const ITextItalicStyle&      other) const;

                bool                                                    isItalic() const;
                void                                                    setItalic(bool  value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextUnderlineStyle
//      A style that determines whether the text it is applied to is underlined or not (true means underlined).
//========================================================================================
class ITextUnderlineStyle : public ICharacterStyle {
        public:
                                                                                ITextUnderlineStyle();
                                                                                ITextUnderlineStyle(bool        isSet);
                                                                                ITextUnderlineStyle(const ITextUnderlineStyle&  other);

                virtual                                                 ~ITextUnderlineStyle();

                ITextUnderlineStyle&                    operator=(const ITextUnderlineStyle&    other);

                bool                                                    operator==(const ITextUnderlineStyle&   other) const;
                bool                                                    operator!=(const ITextUnderlineStyle&   other) const;

                bool                                                    isUnderlined() const;
                void                                                    setUnderlined(bool      value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextStrikethroughStyle
// A style that determines whether the text it is applied to has a strikethrough bar or not (true means struck through).
//========================================================================================
class ITextStrikethroughStyle : public ICharacterStyle {
        public:
                                                                                ITextStrikethroughStyle();
                                                                                ITextStrikethroughStyle(bool    isSet);
                                                                                ITextStrikethroughStyle(const ITextStrikethroughStyle&  other);

                virtual                                                 ~ITextStrikethroughStyle();

                ITextStrikethroughStyle&                operator=(const ITextStrikethroughStyle&        other);

                bool                                                    operator==(const ITextStrikethroughStyle&       other) const;
                bool                                                    operator!=(const ITextStrikethroughStyle&       other) const;

                bool                                                    isStrikethrough() const;
                void                                                    setStrikethrough(bool   value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextOutlineStyle
//      A style that determines whether the text it is applied to is drawn in outline or not (true means outline).
//========================================================================================
class ITextOutlineStyle : public ICharacterStyle {
        public:
                                                                                ITextOutlineStyle();
                                                                                ITextOutlineStyle(bool  isSet);
                                                                                ITextOutlineStyle(const ITextOutlineStyle&      other);

                virtual                                                 ~ITextOutlineStyle();

                ITextOutlineStyle&                              operator=(const ITextOutlineStyle&      other);

                bool                                                    operator==(const ITextOutlineStyle&     other) const;
                bool                                                    operator!=(const ITextOutlineStyle& other) const;

                bool                                                    isOutlined() const;
                void                                                    setOutlined(bool        value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextUneditableStyle
// A style that determines whether the text it is applied to is editable or not (true means NOT editable).
//========================================================================================
class ITextUneditableStyle : public ICharacterStyle {
        public:
                                                                                ITextUneditableStyle();
                                                                                ITextUneditableStyle(bool       isSet);
                                                                                ITextUneditableStyle(const ITextUneditableStyle&        other);

                virtual                                                 ~ITextUneditableStyle();

                ITextUneditableStyle&                   operator=(const ITextUneditableStyle&   other);

                bool                                                    operator==(const ITextUneditableStyle&  other) const;
                bool                                                    operator!=(const ITextUneditableStyle&  other) const;

                bool                                                    isUneditable() const;
                void                                                    setUneditable(bool      value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextBitmapStyle
// A style that determines whether the text it is applied to should be drawn using a
// bitmap font or an outline font (true means bitmap).
//========================================================================================
class ITextBitmapStyle : public ICharacterStyle {
        public:
                                                                                ITextBitmapStyle();
                                                                                ITextBitmapStyle(bool   isSet);
                                                                                ITextBitmapStyle(const ITextBitmapStyle&        other);

                virtual                                                 ~ITextBitmapStyle();

                ITextBitmapStyle&                               operator=(const ITextBitmapStyle&       other);

                bool                                                    operator==(const ITextBitmapStyle&      other) const;
                bool                                                    operator!=(const ITextBitmapStyle&      other) const;

                bool                                                    isBitmap() const;
                void                                                    setBitmap(bool  value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                bool                                                    fValue;
};

//========================================================================================
// CLASS ITextColorStyle
//      A style that controls the color of the text it is applied to.
//========================================================================================
class ITextColorStyle : public ICharacterStyle{
        public:
                                                                                ITextColorStyle();
                                                                                ITextColorStyle(        unsigned char   red,
                                                                                                                        unsigned char   green,
                                                                                                                        unsigned char   blue);
                                                                                ITextColorStyle(const ITextColorStyle&  other);

                virtual                                                 ~ITextColorStyle();

                ITextColorStyle&                                operator=(const ITextColorStyle&        other);

                bool                                                    operator==(const ITextColorStyle&       other) const;
                bool                                                    operator!=(const ITextColorStyle&       other) const;

                void                                                    colors( unsigned char&  red,
                                                                                                unsigned char&  greed,
                                                                                                unsigned char&  blue) const;
                void                                                    setColors(      unsigned char   newRed,
                                                                                                        unsigned char   newGreen,
                                                                                                        unsigned char   newBlue);

                unsigned char                                   red() const;
                unsigned char                                   green() const;
                unsigned char                                   blue() const;

                void                                                    setRed(unsigned char    newValue);
                void                                                    setGreen(unsigned char  newValue);
                void                                                    setBlue(unsigned char   newValue);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                unsigned char                                   fRed;
                unsigned char                                   fGreen;
                unsigned char                                   fBlue;
};

//========================================================================================
// CLASS ITextSuperSubscriptStyle
//      A style that controls the baseline shift of the text it is applied to.
//========================================================================================
class ITextSuperSubscriptStyle : public ICharacterStyle {
        public:
                                                                                ITextSuperSubscriptStyle();
                                                                                ITextSuperSubscriptStyle(float offset);
                                                                                ITextSuperSubscriptStyle(const ITextSuperSubscriptStyle&        other);

                virtual                                                 ~ITextSuperSubscriptStyle();

                ITextSuperSubscriptStyle&               operator=(const ITextSuperSubscriptStyle&       other);

                bool                                                    operator==(const ITextSuperSubscriptStyle&      other) const;
                bool                                                    operator!=(const ITextSuperSubscriptStyle&      other) const;

                float                                                   offset() const;
                void                                                    setOffset(float value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                float                                                   fValue;
};

//========================================================================================
// CLASS ITextFixedPitchStyle
//      A style that controls the Fixed pitch font
//========================================================================================
class ITextFixedPitchStyle : public ICharacterStyle {
        public:
              ITextFixedPitchStyle();
              ITextFixedPitchStyle(bool isSet);
              ITextFixedPitchStyle(const ITextFixedPitchStyle&    other);

              virtual  ~ITextFixedPitchStyle();

              ITextFixedPitchStyle&  operator=(const ITextFixedPitchStyle&     other);

              bool operator==(const ITextFixedPitchStyle& other) const;
              bool operator!=(const ITextFixedPitchStyle& other) const;

              bool isFixedPitch() const;
              void setFixedPitch(bool        value);

        protected:
              virtual void  addToSet(ITextStyleSet& set) const;
              virtual void  removeFromSet(ITextStyleSet&    set) const;
              virtual void  getValueFromSet(const ITextStyleSet&    set);
              virtual bool  isContainedInSet(const ITextStyleSet&   set) const;
              virtual bool  isValueContainedInSet(const ITextStyleSet& set) const;

        private:
              bool  fValue;
};


//========================================================================================
// PARAGRAPH STYLES
//========================================================================================

//========================================================================================
// CLASS IParagraphJustificationStyle
//      A style that controls the justification direction of the paragraphs it is applied to.
//========================================================================================
class IParagraphJustificationStyle : public IParagraphStyle {
        public:
                enum EJustification {
                        kStart,         // align to start of line (left in left-to-right test)
                        kCenter,        // center
                        kEnd,           // align to end of line (right in left-to-right text)

                        kMaxJustification = 127
                };

                                                                                IParagraphJustificationStyle();
                                                                                IParagraphJustificationStyle(EJustification     value);
                                                                                IParagraphJustificationStyle(const IParagraphJustificationStyle&        other);

                virtual                                                 ~IParagraphJustificationStyle();

                IParagraphJustificationStyle&   operator=(const IParagraphJustificationStyle&   other);

                bool                                                    operator==(const IParagraphJustificationStyle&  other) const;
                bool                                                    operator!=(const IParagraphJustificationStyle&  other) const;

                EJustification                                  justification() const;
                void                                                    setJustification(EJustification value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                EJustification                                  fValue;
};

//========================================================================================
// CLASS IParagraphSpreadStyle
//      A style that controls which lines of the paragraphs it is applied to (if any) are fully justified (i.e., aligned to both the left and right margins).
//========================================================================================
class IParagraphSpreadStyle : public IParagraphStyle {
        public:
                enum ESpread {
                        kNone,                  // never fully justify (ragged edges)
                        kAll,                   // justify all lines
                        kNotLastLine,   // justify all but the last line of each paragraph

                        kMaxSpread = 127
                };

                                                                                IParagraphSpreadStyle();
                                                                                IParagraphSpreadStyle(ESpread   value);
                                                                                IParagraphSpreadStyle(const IParagraphSpreadStyle&      other);

                virtual                                                 ~IParagraphSpreadStyle();

                IParagraphSpreadStyle&                  operator=(const IParagraphSpreadStyle&  other);

                bool                                                    operator==(const IParagraphSpreadStyle& other) const;
                bool                                                    operator!=(const IParagraphSpreadStyle& other) const;

                ESpread                                                 spread() const;
                void                                                    setSpread(ESpread       value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                ESpread                                                 fValue;
};

//========================================================================================
// CLASS IParagraphStartIndentStyle
//      A style that controls the starting indent distance of the paragraphs it is applied to.
//========================================================================================
class IParagraphStartIndentStyle : public IParagraphStyle {
        public:
                                                                                IParagraphStartIndentStyle();
                                                                                IParagraphStartIndentStyle(float        value);
                                                                                IParagraphStartIndentStyle(const IParagraphStartIndentStyle&    other);

                virtual                                                 ~IParagraphStartIndentStyle();

                IParagraphStartIndentStyle&             operator=(const IParagraphStartIndentStyle&     other);

                bool                                                    operator==(const IParagraphStartIndentStyle&    other) const;
                bool                                                    operator!=(const IParagraphStartIndentStyle&    other) const;

                float                                                   startIndent() const;
                void                                                    setStartIndent(float    value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                float                                                   fValue;
};

//========================================================================================
// CLASS IParagraphEndIndentStyle
//      A style that controls the ending indent distance of the paragraphs it is applied to.
//========================================================================================
class IParagraphEndIndentStyle : public IParagraphStyle {
        public:
                                                                                IParagraphEndIndentStyle();
                                                                                IParagraphEndIndentStyle(float  value);
                                                                                IParagraphEndIndentStyle(const IParagraphEndIndentStyle&        other);

                virtual                                                 ~IParagraphEndIndentStyle();

                IParagraphEndIndentStyle&               operator=(const IParagraphEndIndentStyle&       other);

                bool                                                    operator==(const IParagraphEndIndentStyle&      other) const;
                bool                                                    operator!=(const IParagraphEndIndentStyle&      other) const;

                float                                                   endIndent() const;
                void                                                    setEndIndent(float      value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                float                                                   fValue;
};

//========================================================================================
// CLASS IParagraphFirstIndentStyle
//      A style that controls the first-line indent distance of the paragraphs it is applied to.
//========================================================================================
class IParagraphFirstIndentStyle : public IParagraphStyle {
        public:
                                                                                IParagraphFirstIndentStyle();
                                                                                IParagraphFirstIndentStyle(float        value);
                                                                                IParagraphFirstIndentStyle(const IParagraphFirstIndentStyle&    other);

                virtual                                                 ~IParagraphFirstIndentStyle();

                IParagraphFirstIndentStyle&             operator=( const IParagraphFirstIndentStyle&    other);

                bool                                                    operator==(const IParagraphFirstIndentStyle&    other) const;
                bool                                                    operator!=(const IParagraphFirstIndentStyle&    other) const;

                float                                                   firstIndent() const;
                void                                                    setFirstIndent(float    value);

        protected:
                virtual void                                    addToSet(ITextStyleSet& set) const;
                virtual void                                    removeFromSet(ITextStyleSet&    set) const;
                virtual void                                    getValueFromSet(const ITextStyleSet&    set);
                virtual bool                                    isContainedInSet(const ITextStyleSet&   set) const;
                virtual bool                                    isValueContainedInSet(const ITextStyleSet&      set) const;

        private:
                float                                                   fValue;
};

//========================================================================================
// CLASS ITabRulerStyle
//      A style that controls the tab stops of the paragraphs it is applied to.
//========================================================================================
class ITabRulerStyle : public IParagraphStyle {
public:

                                ITabRulerStyle(const ITabRulerStyle& source);
                                ITabRulerStyle(GCoord autoSpacing);
                                ITabRulerStyle(size_t tabCount, const ITabStop* tabs, GCoord autoSpacing);
                                ITabRulerStyle(const IMTabRuler& ruler);
        virtual         ~ITabRulerStyle();

        void            setRuler(const IMTabRuler& ruler);

        IMTabRuler* ruler();

protected:
                                                        ITabRulerStyle();

        virtual void                    addToSet(ITextStyleSet& set) const;
        virtual void                    removeFromSet(ITextStyleSet& set) const;
        virtual void                    getValueFromSet(const ITextStyleSet& set);
        virtual bool                    isContainedInSet(const ITextStyleSet& set) const;
        virtual bool                    isValueContainedInSet(const ITextStyleSet& set) const;

private:
        IMTabRuler*                     fRuler;
};

#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _ISTYLES_
