/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 83 1.12.1.4 source/core/streams/istrmio.hpp, oss, ioc.v400, 980918 
//
//
//
// istrmio.hpp
//
#ifndef _ISTRMIO_
#define _ISTRMIO_

#include <iprimtyp.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

//
// external class declarations.
//
class IMStreamable;
class IMRefCounted;
class ITypeRepresentation;
class IStreamFinder;
class ITypeName;
class IStreamContextFrame;

#pragma pack(push,4)
#pragma enum(4)

//
// IStreamIO
//
class IStreamIO
{
protected:
        IStreamIO();
        IStreamIO(const IStreamIO& other);
        virtual ~IStreamIO();

friend class IDataStream;
        //
        // called from the IDataStream constructor so that
        // the IStreamIO can stream subobjects.
        //
        virtual void    setStream(IDataStream* dataStream) = 0;

        //
        // reset the stream back to the beginning.
        //
        virtual void    reset() = 0;

        //
        // force all data out through the stream buffer.
        //
        virtual void    flush() = 0;

        //
        // write primitive data types.  encoding format is implemented by
        // subclasses.
        //
        virtual void    writeByte(char source) = 0;
        virtual void    writeShort(short source) = 0;
        virtual void    writeLong(long source) = 0;
        virtual void    writeFloat(float source) = 0;
        virtual void    writeDouble(double source) = 0;

        //
        // read primitive data types.
        //
        virtual char    readByte() = 0;
        virtual short   readShort() = 0;
        virtual long    readLong() = 0;
        virtual float   readFloat() = 0;
        virtual double  readDouble() = 0;

        //
        // support for transfers of arrays.
        //
        virtual void    writeBytes(const char* data, unsigned long howMany) = 0;
        virtual void    readBytes(char* data, unsigned long howMany) = 0;
        virtual void    writeShorts(const short* data, unsigned long howMany) = 0;
        virtual void    readShorts(short* data, unsigned long howMany) = 0;
        virtual void    writeLongs(const long* data, unsigned long howMany) = 0;
        virtual void    readLongs(long* data, unsigned long howMany) = 0;
        virtual void    writeFloats(const float* data, unsigned long howMany) = 0;
        virtual void    readFloats(float* data, unsigned long howMany) = 0;
        virtual void    writeDoubles(const double* data, unsigned long howMany) = 0;
        virtual void    readDoubles(double* data, unsigned long howMany) = 0;

friend class IStreamableGlobalTemplateHelper;
        //
        // support for streamable objects.
        //
        virtual void    writeObject(const IMStreamable* object, const ITypeRepresentation* typeRep) = 0;
        virtual IMStreamable*   readObject(const IStreamFinder* finder, const ITypeRepresentation* typeRep) = 0;
        virtual bool    writeAliasedObject(const IMStreamable* object, const IMRefCounted* refCounted, const ITypeRepresentation* typeRep) = 0;
        virtual bool    readAliasedObject(IMStreamable*& object, IMRefCounted*& refCounted, const IStreamFinder* finder, const ITypeRepresentation* typeRep) = 0;

friend class IStreamInFrame;
friend class IStreamOutFrame;
        //
        // support for stream frames.
        //
        virtual void    writeBeginFrame() = 0;
        virtual void    writeEndFrame() = 0;
        virtual void    readBeginFrame() = 0;
        virtual void    readEndFrame() = 0;
        virtual bool    atEndOfFrame() = 0;

friend class IStreamContextFrame;
        //
        // support for stream context.
        //
        virtual bool    establishContext() = 0;
        virtual void    clearContext() = 0;

        //
        // support for subclasses.
        //
        static  IMStreamable*   createUninitialized(const IStreamFinder* finder, const ITypeName& typeName, IMRefCounted*& refCounted);
        static  const ITypeRepresentation*      findTypeRepresentation(const IStreamFinder* finder, const ITypeName& typeName);
};

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _ISTRMIO_
