/**
 * This file describes the 2-D point array used to store and manipulate linear arrays of points.
 * Classes described here are IGPoint2DArray and IGRPoint2DArray.
 *
 *   IBM Open Class Library
 *   (C) Copyright International Business Machines Corporation,  1997
 *   Licensed Material - Program-Property of IBM - All Rights Reserved.
 */

// Revision: 38 1.15.1.6 source/albert/graph2d/ipt2darr.hpp, 2d, ioc.v400, 980918 

#ifndef _IPT2DARR_
#define _IPT2DARR_

#include <irawarr.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

class IGRect2D;
class IGPoint2D;
class IGRPoint2D;

#pragma pack(push,4)
#pragma enum(4)

/**
 * IGPoint2DArray provides array access for collections of IGPoints.
 * IGPoint2DArray provides array access for collections of IGPoints.
 * This is used by IGPolyline2D, IGPolygon2D, etc.
 * Eventually when we get templates it may become a subclass of a generic array class.
 * Note: IRawArray does not check whether allocation is successful. Before constructing
 * a new array, one should check if there is enough memory and use a reasonable size.
 */

class IGPoint2DArray : public IRawArray<IGPoint2D>
{
  public:

	/**
	 * Creates an array of IGPoints of the specified size. The points are left uninitialized.
	 */
	IGPoint2DArray(unsigned long size = 0);

	/**
	 * Copy constructor.
	 */
	IGPoint2DArray(const IGPoint2DArray& source);

	~IGPoint2DArray();

	/**
	 * Gets the IGPoint2D at the specified index.
	 */
	IGPoint2D point(unsigned long i) const;

	/**
	 * Sets the IGPoint2D at the specified index, to the specified point.
	 */
	void setPoint(unsigned long i, const IGPoint2D& p);

	/**
	 * Computes the number of points in the IGPoint2DArray.
	 */
	unsigned long numberOfPoints() const;

	/**
	 * Creates an IGRect2D that bounds the points in the IGPoint2DArray.
	 */
	IGRect2D bounds() const;

	IDataStream& operator>>=(IDataStream&) const;
	IDataStream& operator<<=(IDataStream&);
};


/**
 * IGRPoint2DArray provides array access for collections of IGRPoints.
 * IGRPoint2DArray provides array access for collections of IGRPoints.
 * This used by IGCurve2D. Eventually when we get templates it may become a subclass of a generic
 * arry class.
 * Note: IRawArray does not check whether allocation is successful. Before constructing
 * a new array, one should check if there is enough memory and use a reasonable size.
 */

 class IGRPoint2DArray : public IRawArray<IGRPoint2D> {
  public:

	/**
	 * Creates an array of IGRPoints of the specified size. The points are left uninitialized.
	 */
	IGRPoint2DArray(unsigned long size = 0);

	/**
	 * Constructs an IGRPoint2DArray from an IGPoint2DArray by setting fW = 1.0.
	 */
	IGRPoint2DArray(const IGPoint2DArray& source);  // Sets w's to 1.0

	/**
	 * Copy constructor
	 */
	IGRPoint2DArray(const IGRPoint2DArray& source);

	~IGRPoint2DArray();
	
	/**
	 * Gets the IGRPoint2D at the specified index.
	 */
	IGRPoint2D point(unsigned long i) const;

	/**
	 * Sets the IGRPoint2D at the specified index, to the specified point.
	 */
	void setPoint(unsigned long i, const IGRPoint2D& p);

	/**
	 * Computes the number of points in the IGRPoint2DArray.
	 */
	unsigned long numberOfPoints() const;

	/**
	 * Creates a IGRecte2D that bounds the points in the IGRPoint2DArray.
	 */
	IGRect2D bounds() const;

	IDataStream& operator>>= (IDataStream&) const;
	IDataStream& operator<<= (IDataStream&);
};

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#include <ipt2darr.inl>

#endif // _IPT2DARR_
