/**
 * This file describes paint attributes needed for rendering.
 * Classes described here are IMaskPattern and IPaint
 *
 *   IBM Open Class Library
 *   (C) Copyright International Business Machines Corporation,  1997
 *   Licensed Material - Program-Property of IBM - All Rights Reserved.
 */

// Revision: 15 1.26.1.6 source/albert/graph2d/ipaints.hpp, 2d, ioc.v400, 980918 

#ifndef _IPAINTS_
#define _IPAINTS_

#include <igbase2d.hpp>
#include <ibcolor.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

class IGPoint2D;
class IGImage;

#include <igrtypes.hpp>
#include <imstrmbl.hpp>

#pragma pack(push,4)
#pragma enum(4)

/**
 * IMaskPattern is an 8x8 image pattern of 1-bit color depth (black = transparent, white = solid).
 */

class IMaskPattern
{
public:

	/**
	 * Constructs a custom pattern.
	 */
	IMaskPattern(
		unsigned char row0,
		unsigned char row1,
		unsigned char row2,
		unsigned char row3,
		unsigned char row4,
		unsigned char row5,
		unsigned char row6,
		unsigned char row7);

	/**
	 * Copy constructor.
	 */
	IMaskPattern(const IMaskPattern& source);

	/**
	 * Assignment operator.
	 */
	IMaskPattern& operator=(const IMaskPattern& source);

	~IMaskPattern();

	/**
	 * Tests two IMaskPattern objects for equality.
	 */
	bool operator==(const IMaskPattern&) const;

	/**
	 * Tests two IMaskPattern objects for inequality.
	 */
	bool operator!=(const IMaskPattern&) const;

	/** Predefined patterns */
    static const IMaskPattern& solid();
    static const IMaskPattern& diagonalDown();
    static const IMaskPattern& cross();
    static const IMaskPattern& diagonalCross();
    static const IMaskPattern& diagonalUp();
    static const IMaskPattern& horizontal();
    static const IMaskPattern& vertical();

	IDataStream& operator>>=(IDataStream& toWhere) const;
	IDataStream& operator<<=(IDataStream& fromWhere);

    private:
friend class IPaint;
friend class IGrafDevice;
friend class IGraph2DTestHelper;
friend class IMaskPatternStatics;
	// using int instead of private IGrafDevice::EMaskPattern
	IMaskPattern(int pattern);
	int pattern() const;
	const unsigned char* bits() const;
private:
	int fPattern;
#ifndef IC_MOTIF
	unsigned short* fBits; // unsigned short [8] if allocated
#else
	unsigned char* fBits; // unsigned char [8] if allocated
#endif
};

/**
 * IPaint is a concrete class that encapsulates the color and image pattern the user can set.
 * IPaint is a concrete class that encapsulates the color and image pattern the user can set.
 * It is used to specify the color appearance of the source primitive.
 */

class IPaint : public IMSTREAMABLE
{
public:

	/**
	 * Mask pattern is set to solid and pattern phase is set to zero.
	 * Hit flag is set to false and hit detection is enabled.
	 * Moved from protected for streaming.
	 */
    IPaint();

    virtual ~IPaint();

	/**
	 */
    IPaint(const IPaint&);

	/**
	 */
	IPaint(
		const IBaseColor& aColor,
		const IMaskPattern& maskPattern = IMaskPattern::solid(),
		const IGPoint2D& patternPhase = IGPoint2D::origin());

	/**
	 */
	IPaint(
		const IGImage& imagePattern,
		const IGPoint2D& patternPhase = IGPoint2D::origin());

	/**
	 * Returns the color of the IPaint.
	 */
    virtual const IBaseColor* color() const;

	/**
	 * Returns the color of the IPaint.
	 */
    virtual const IMaskPattern* maskPattern() const;

	/**
	 * Returns offset point to start tiling of the IPaint.
	 */
    virtual const IGPoint2D* patternPhase() const;

	/**
	 * Returns the image pattern of the IPaint.
	 */
    virtual const IGImage* imagePattern() const;

	/**
	 * Sets the color of the paint after deleting the existing one.
	 */
    virtual void setColor(const IBaseColor& color);

	/**
	 * Sets the tiling of the paint after deleting the existing one.
	 */
    virtual void setMaskPattern(const IMaskPattern& MaskPattern = IMaskPattern::solid());

	/**
	 * Sets the offset point to start tiling of the paint.
	 */
    virtual void setPatternPhase(const IGPoint2D& patternPhase);

	/**
	 * Sets the image pattern of the paint after deleting the existing one.
	 */
    virtual void setImagePattern(const IGImage& imagePattern);

	/**
	 * Deleting the image pattern and resets to NIL.
	 */
    virtual void removeImagePattern();

	/**
	 * Assignment operator.
	 */
    IPaint& operator=(const IPaint& source);

	/**
	 * Tests two IPaint objects for equality.
	 */
	virtual bool operator==(const IPaint&) const;

	/**
	 * Tests two IPaint objects for inequality.
	 */
	virtual bool operator!=(const IPaint&) const;

	StreamableDeclarationsMacro(IPaint);

protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:
    IBaseColor fBaseColor;
    IBaseColor* fColor;
    const IMaskPattern* fMaskPattern;
    IMaskPattern* fMaskPatStorage;
    IGPoint2D fPatternPhase;
    IGImage* fImage;
};

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _IPAINTS_
