/**
 * Generic exception classes thrown by the low levels of OpenClass:
 * <ul>
 *      <li>IObjectNotFound     - Unable to locate an object (e.g. file, net resource, etc.)
 *      <li>IInvalidName        - Name was invalid (e.g. path name, file name, etc.)
 *      <li>ICannotProceed      - Operation was not completed but may be continued later
 *      <li>IAlreadyExists      - Attempt to create something that already exists
 *      <li>IMustBeEmpty        - Invalid operation (e.g. delete) on a non-empty container
 * </ul>
 *      All of these exception classes have constructors of the form
 *        <code>IMyExceptionClass(const char *text = 0);</code>
 *
 *      These constructors allow you to pass in a customized error message string.
 *      If you use the default value of 0, you will get a generic error message.
 *
 * @author Mike Seilnacht, Laura Werner
 *
 * @package Utilities
 *
 * @copyright
 *      IBM Open Class Library
 *      (C) Copyright International Business Machines Corporation 1992-1997
 *      Licensed Material - Program-Property of IBM - All Rights Reserved.
 *
 * Revision:   13   1.15.1.4   source/core/utils/igenexc.hpp, filesystem, ioc.v400, 980918   
 */
#ifndef _IGENEXC_
#define _IGENEXC_

#include <iexcbase.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

// The definition of ITHROW in the OS2 version of iexcbase.hpp is wrong
// Provide a correct one here until the real one is fixed
#ifdef __OS2__
  #undef ITHROW
  #define ITHROW(exc)\
     exc.addLocation(IEXCEPTION_LOCATION()),\
     exc.logExceptionData(),\
     throw(exc)
#endif

#pragma pack(push,4)
#pragma enum(4)

/** ------------------------------------------------------------------------
 * An exception indicating that a requested object could not be created
 * because an object with the same name or location already exists.
 */
class IAlreadyExists : public IAccessError {
public:
        /**
         * Constructs an IAlreadyExists exception with an optional
         * customized error message.
         *
         * @param text  The error message to use for this exception.  If no
         *                              error message is provided, a default message will
         *                              be used instead.
         */
                                                IAlreadyExists(const char* text = 0);

                                                IAlreadyExists(const IAlreadyExists &);
        virtual                         ~IAlreadyExists();
        virtual const char*     name() const;

private:
  IAlreadyExists&               operator = ( IAlreadyExists & );
};

/** ------------------------------------------------------------------------
 * An exception indicates that an operation could not be completed,
 * but that it may be possible to continue the operation or recover in
 * some other fashion.
 */
class ICannotProceed : public IException {
public:
        /**
         * Constructs an ICannotProceed exception with an optional
         * customized error message.
         *
         * @param text  The error message to use for this exception.  If no
         *                              error message is provided, a default message will
         *                              be used instead.
         */
                                                ICannotProceed(const char* text = 0);

                                                ICannotProceed(const ICannotProceed &);
        virtual                         ~ICannotProceed();
        virtual const char*     name() const;

private:
  ICannotProceed&               operator = ( ICannotProceed & );
};

/** ------------------------------------------------------------------------
 * An exception indicating that an invalid name for an object (e.g. file,
 * network resource) was used.
 */
class IInvalidName : public IInvalidRequest {
public:
        /**
         * Constructs an IInvalidName exception with an optional
         * customized error message.
         *
         * @param text  The error message to use for this exception.  If no
         *                              error message is provided, a default message will
         *                              be used instead.
         */
                                                IInvalidName(const char* text = 0);

                                                IInvalidName(const IInvalidName &);
        virtual                         ~IInvalidName();
        virtual const char*     name() const;

private:
  IInvalidName&                 operator = ( IInvalidName & );
};

/** ------------------------------------------------------------------------
 * An exception indicating that an operation on a container failed because
 * the container was not empty.
 */
class IMustBeEmpty : public IAccessError {
public:
        /**
         * Constructs an IMustBeEmpty exception with an optional
         * customized error message.
         *
         * @param text  The error message to use for this exception.  If no
         *                              error message is provided, a default message will
         *                              be used instead.
         */
                                                IMustBeEmpty(const char* text = 0);

                                                IMustBeEmpty(const IMustBeEmpty &);
        virtual                         ~IMustBeEmpty();
        virtual const char*     name() const;

private:
  IMustBeEmpty&                 operator = ( IMustBeEmpty & );
};

/** ------------------------------------------------------------------------
 * An exception indicating that an operation failed because it
 * was unable to locate the requested object.
 */
class IObjectNotFound : public IAccessError {
public:
        /**
         * Constructs an IObjectNotFound exception with an optional
         * customized error message.
         *
         * @param text  The error message to use for this exception.  If no
         *                              error message is provided, a default message will
         *                              be used instead.
         */
                                                IObjectNotFound(const char* text = 0);

                                                IObjectNotFound(const IObjectNotFound &);
        virtual                         ~IObjectNotFound();
        virtual const char*     name() const;

private:
  IObjectNotFound&              operator = ( IObjectNotFound & );
};

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _IGENEXC_
