// ----------------------------------------------------------------------------
// FILE NAME: idbcsbuf.hpp
//
// DESCRIPTION:
//   Declaration of the classes:
//     IDBCSBuffer
//
// COPYRIGHT:
//   IBM Open Class Library
//   Licensed Materials - Property of IBM
//
//   5645-001
//   (C) Copyright IBM Corporation 1992, 1997  All Rights Reserved.
//
// Revision: 02  1.5.1.4  source/core/base/idbcsbuf.hpp, appsupport, ioc.v400, 980918  
// ----------------------------------------------------------------------------

#ifndef _IDBCSBUF_
#define _IDBCSBUF_

#include <ibuffer.hpp>

extern "C" {
#include <stdlib.h>
#include <limits.h>
#include <ctype.h>
#include <string.h>
}

#ifdef IC_NLS

#ifdef _MSC_VER
    // MSVC++ wants wchar.h included without extern "C"
    #include <wchar.h>
#else
    extern "C"
    {
        #include <wchar.h>
    }
#endif

#if defined(IC_PMWIN) || defined(IC_MVS400)
    extern "C"
    {
        #include <wctype.h>
    }
#endif
#endif

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma pack(push,4)
#pragma enum(4)

class IC_EXPORTB IDBCSBuffer : public IBuffer
{
public:

#ifdef IC_NLS
    // ---------------------------- Comparisons -------------------------------
    //  See ibuffer.hpp for definition of Comparison  */
    virtual Comparison compare(const void* p, unsigned len ) const;
#endif

    // ------------------------- Subset ---------------------------------------
    virtual IDBCSBuffer* subString (unsigned startPos,
                                    unsigned len,
                                    char     padCharacter ) const;

#ifdef IC_NLS
    virtual IDBCSBuffer* subString(unsigned startPos) const;

    // --------------------------------- Testing ------------------------------
    virtual bool isAlphanumeric() const;
    virtual bool isAlphabetic() const;
    virtual bool isASCII() const;
    virtual bool isControl() const;
    virtual bool isDigits() const;
    virtual bool isGraphics() const;
    virtual bool isHexDigits() const;
    virtual bool isLowerCase() const;
    virtual bool isPrintable() const;
    virtual bool isPunctuation() const;
    virtual bool isUpperCase() const;
    virtual bool isWhiteSpace() const;
#endif

    // ------------------------- NLS Testing ----------------------------------
    virtual bool isMBCS() const;
    virtual bool isDBCS() const;
    virtual bool isSBCS() const;
    virtual bool isValidMBCS() const;
    virtual bool isValidDBCS() const;
    virtual bool includesMBCS() const;
    virtual bool includesDBCS() const;
    virtual bool includesSBCS() const;

    // ------------------------- Searches -------------------------------------
    virtual unsigned
    indexOf           ( const IStringTest&  aTest,
                        unsigned            startPos = 1 ) const,
    indexOf           ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 1 ) const,
    indexOfAnyBut     ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 1 ) const,
    indexOfAnyBut     ( const IStringTest&  aTest,
                        unsigned            startPos = 1 ) const,
    indexOfAnyOf      ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 1 ) const,
    indexOfAnyOf      ( const IStringTest&  aTest,
                        unsigned            startPos = 1 ) const,
    lastIndexOf       ( const IStringTest&  aTest,
                        unsigned            startPos = 1 ) const,
    lastIndexOf       ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 0 ) const,
    lastIndexOfAnyBut ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 0 ) const,
    lastIndexOfAnyBut ( const IStringTest&  aTest,
                        unsigned            startPos = 0 ) const,
    lastIndexOfAnyOf  ( const char*         pString,
                        unsigned            len,
                        unsigned            startPos = 0 ) const,
    lastIndexOfAnyOf  ( const IStringTest&  aTest,
                        unsigned            startPos = 0 ) const;


/*------------------------- Editing ------------------------------------------*/
    virtual IDBCSBuffer* center(unsigned newLen, char padCharacter );

#ifdef IC_NLS
    virtual IDBCSBuffer* change(const char* pSource,
                                unsigned    sourceLen,
                                const char* pTarget,
                                unsigned    targetLen,
                                unsigned    startPos,
                                unsigned    numChanges );
#endif

    virtual IDBCSBuffer* insert(const char *pInsert,
                                unsigned    insertLen,
                                unsigned    pos,
                                char        padCharacter );

    virtual IDBCSBuffer* leftJustify(unsigned newLen, char padCharacter );
    virtual IDBCSBuffer* lowerCase();
    virtual IDBCSBuffer* overlayWith(const char*    overlay,
                                    unsigned        len,
                                    unsigned        pos,
                                    char            padCharacter );
    virtual IDBCSBuffer* remove(unsigned startPos, unsigned numChars );
    virtual IDBCSBuffer* reverse();
    virtual IDBCSBuffer* rightJustify(unsigned newLen, char padCharacter );
    virtual IDBCSBuffer* strip( const char*             pChars,
                                unsigned                len,
                                IStringEnum::StripMode  mode );
    virtual IDBCSBuffer* strip( const IStringTest&      aTest,
                                IStringEnum::StripMode  mode );
    virtual IDBCSBuffer* translate( const char* pInputChars,
                                    unsigned    inputLen,
                                    const char* pOutputChars,
                                    unsigned    outputLen,
                                    char        padCharacter );
    virtual IDBCSBuffer* upperCase( );

    // ------------------------- Allocation -----------------------------------
    virtual IDBCSBuffer* allocate(unsigned newLen) const;

    // ------------------------- Queries --------------------------------------
    virtual IStringEnum::CharType charType(unsigned index) const;

    virtual char* next(const char* prev);

    virtual const char* next(const char* prev) const;

protected:
    // ------------------------ Search Initialization -------------------------
    virtual unsigned startSearch(unsigned startPos, unsigned searchLen) const;
    virtual unsigned startBackwardsSearch(unsigned startPos, unsigned searchLen) const;


    // ------------------------- Protected Testing ----------------------------
    bool isCharValid(   unsigned    pos,
                        const char* pValidChars,
                        unsigned    numValidChars ) const;






    static bool isSBC(char const* pChar);

    // ------------------------- Protected Queries ----------------------------
    const char* className() const;

    static size_t charLength(char const* pChar);
    static size_t maxCharLength();

    size_t charLength(unsigned pos) const;
    size_t prevCharLength(unsigned pos) const;


    friend class IBuffer;

    // ------------------------- Protected Constructors -----------------------
    IDBCSBuffer(unsigned bufLength);
    ~IDBCSBuffer();


    // ------------------------- Protected New Functions ----------------------
    // These functions are for Internal use only 
#ifdef IC_NLS
#ifdef IC_HAS_RESTARTABLE

    static bool isSBC(char const* pChar , mbstate_t* pMBState);
    static size_t charLength(char const* pChar, mbstate_t* pMBState);

    size_t charLength(unsigned pos, mbstate_t* pMBState) const;
    size_t prevCharLength(unsigned pos, mbstate_t* pMBState) const;
#endif

private:
    unsigned windexOfAnyBut(const IStringTest& aTest, unsigned startPos = 1) const;
#endif

};

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#include <idbcsbuf.inl>

#endif // _IDBCSBUF_




