/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 79 1.11.1.18 source/albert/graph2d/icolmap.hpp, 2d, ioc.v400, 980918 
/*==========================================
||
||      File:    icolmap.hpp
||
||  Class:  IRGBAColorArray
||          IColorMap
||          IColorMapContext
||
||      Change History:
||
*/

#ifndef _ICOLMAP_
#define _ICOLMAP_

#include <ibcolor.hpp>
#include <i2dghand.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma pack(push, 4)
#pragma enum(4)

//There is no such class, but declared for future enhancements
class IColorMapData;

struct IR8G8B8A8Color {
public:
        CharIntensity fRed;
        CharIntensity fGreen;
        CharIntensity fBlue;
        CharIntensity fOpacity;

        operator IBaseColor() const;
};

#if 1 //SAS
class IRGBAColorArray {
public:

        /**
         * Constructs a color array with the specified number of colors.
         */
        IRGBAColorArray(unsigned long numberOfColors = 256);

        /**
         * Copy constructor.
         */
        IRGBAColorArray(const IRGBAColorArray& source);

        virtual ~IRGBAColorArray();

        /**
         * Returns the color of the specified index in the color array.
         */
        IBaseColor color(unsigned long i) const;

        /**
         * Sets the color of the specified index in the color array.
         */
        virtual void setColor(unsigned long i, const IBaseColor& p);

        /**
         * Returns the size of the color array.
         */
        unsigned long numberOfColors()const;

        /**
         * Assignment operator.
         */
        IRGBAColorArray& operator=(const IRGBAColorArray& Src);

        /**
         * Tests the two color arrays for equality.
         */
        virtual bool operator==(const IRGBAColorArray&) const;

        /**
         * Index operator to facilitate access to the data structure by subclasses.
         */
        IR8G8B8A8Color& operator[](unsigned long i);

        /**
         * Index operator to facilitate access to the data structure by subclasses.
         */
        const IR8G8B8A8Color& operator[](unsigned long i) const;

        /**
         * Assume ownership of the allocation of a new color array data allocation.
         */
        virtual void adoptColorArray(unsigned long size, IR8G8B8A8Color* colorArrayToAdopt);

        /**
         * Relinquishes knowledge of, and responsibility for, the color array, returning it to the caller.
         */
        virtual IR8G8B8A8Color* orphanColorArray();

        IDataStream& operator>>=(IDataStream& toWhere) const;
        IDataStream& operator<<=(IDataStream& fromWhere);

protected:
        //uses the IR8G8B8A8Color data directly
        friend class IPixelBuffer;
        friend class IImageConverter;
        friend class IGIFConverter;
        friend class IBMPConverter;
        friend class IJPGConverter;
        friend class IXPMConverter;

        IR8G8B8A8Color* fColors;
        unsigned long fNumberOfColors;

};

#endif //SAS

class IColorMap {
public:

        enum EColorMapType {
                kSystem=-1,
                kTrueColor=0,
                kUserDefined=1
        };
        /**
         * default Construct
         */
        IColorMap( EColorMapType colorMapType = kUserDefined );

        /**
         * Constructs a color map with a color array
         */
        IColorMap(unsigned long, const IR8G8B8A8Color*);

        /**
         * Copy constructor.
         */
        IColorMap(const IColorMap&);


        virtual ~IColorMap();

        /**
         * Assignment operator.
         */
        IColorMap& operator=(const IColorMap&);

        /**
         * Tests the two color arrays for equality.
         */
        virtual bool operator==(const IColorMap&) const;

        /**
         * Test whether a color is a system color. Not implemented yet.
         */
        virtual bool isSystemColor(unsigned long) const;

        /**
         * Get the index of a color in the color map.
         */
        unsigned long colorIndex(const IBaseColor&) const;

        /**
         * Tests if the color index is allocated.
         */
        bool isColorIndexAllocated(unsigned long index) const;

        /**
         * Get the rgb of a color in the system color map given the index.
         */
        static unsigned long systemColorRGB(const long);

        /**
         * Get the index of a color in the system color map.
         */
        static unsigned long systemColorIndex(const IBaseColor&);

        /**
         * Map colors in the color array.
         */
        unsigned long mapColors(unsigned long, const IR8G8B8A8Color*);

        /**
         * Release all cells other than Reserved, return # of released cells
         */
        unsigned long releaseCells();


        enum EColorCellStatus {
                kNotFound=-1,
                kReserved=0,
                kFree=1,
                kUsed=2
        };

        /**
         * does system support colormap
         */
        static bool hasColorMapSupport();

        /**
         * can colormap be modified 
         */
        static bool canChangeColorMap();

        /**
         * get a colormap type with system colormap
         */
        static IColorMap& defaultSystemColorMap();

        /**
         * get a colormap type with true colormap
         */
        static IColorMap& trueColorMap();

        /**
         * load a default web safe color map
         */
        static IColorMap& webSafeColorMap();

        /**
         * get the native color map
         * HPAL for OS/2 and Colormap for AIX
         */
        unsigned long nativeColorMap();

        /**
         * get the color map type
         */
        EColorMapType colorMapType() const;

        /**
        * set the default colormap
        */
        static void setDefaultColorMap(IColorMap *);

        /**
        * get the default colormap
        */
        static IColorMap& defaultColorMap();

        /**
         * set the color map type
         */
        void setColorMapType(EColorMapType);

        /**
         * Returns the size of the color array.
         */
        unsigned long numberOfColors()const;

        /**
         * Index operator to facilitate access to the data structure by
subclasses.
         */
        const IR8G8B8A8Color& operator[](unsigned long) const;

        IDataStream& operator>>=(IDataStream&) const;
        IDataStream& operator<<=(IDataStream&);

private:
        EColorCellStatus colorExists(int, int, int, int *);
        int insertColor(int, int, int, int);

        //uses the IR8G8B8A8Color data directly
        friend class IPixelBuffer;
#ifdef IC_MOTIF
        friend class IXDevice;
#endif //IC_MOTIF

        unsigned long fNumberOfColors;
        unsigned long fNumberOfFreeCells;
        IR8G8B8A8Color* fColors;
        EColorCellStatus *fCellStatus;

        friend class IColorMapContext;

        /**
         * create a platform specific native palette using IR8G8B8A8Color
data
         */

        //fData are HPALETTE in Win, HPAL in os2, Colormap in AIX
        unsigned long fData;
        EColorMapType fColorMapType;

	//There is no such class, but declared for future enhancements
	IColorMapData* fColorMapData;

        static IColorMap* fDefaultColorMap;
        static IColorMap* DefaultSystemColorMap;
        static IColorMap* TrueColorMap;
        static IColorMap* WebSafeColorMap;
};


#if (IC_OBSOLETE <= IC_OBSOLETE_3)
//The class IColorMapContext was inadvertently made public
//in the portapak release.  Its function is superceded by
//the function IFrameWindow::adoptColorMap and so the class
//should be obsoleted.

//------------------------------------------------------------------------
// In the context IWondow, there should be a context class for the window
// independent of the window, because same colormap may be set on different
// windows.

class IColorMapContext {
public:
        IColorMapContext(const INativeWindowHandle& , IColorMap* );
        IColorMapContext(const IPresSpaceHandle& , IColorMap* );
        IColorMap& colormap() const;
        ~IColorMapContext();

private:
        IColorMap* fMap;
        IPresSpaceHandle fPS;
        INativeWindowHandle fWindow;
        bool fReleaseDC;

        unsigned long  fPrevious;  //HPALETTE in Win, HPAL in os2, Colormap in AIX

};
#endif //(IC_OBSOLETE <= IC_OBSOLETE_3)

#pragma enum(pop)
#pragma pack(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _ICOLMAP_

