/**
 * This file describes bundles which are used to collect together the attributes needed for rendering.
 * Classes described here are:
 *  IAttributeState, IFillBundle, IFrameBundle, IFillAndFrameBundle, and IGrafBundle
 *
 *   IBM Open Class Library
 *   (C) Copyright International Business Machines Corporation,  1997
 *   Licensed Material - Program-Property of IBM - All Rights Reserved.
 *
 */

// Revision: 40  1.28.1.7  source/albert/graph2d/ibundles.hpp, 2d, ioc.v400, 980918  

#ifndef _IBUNDLES_
#define _IBUNDLES_

#include <igrtypes.hpp>
#include <igrafatt.hpp>
#include <ipaints.hpp>
#include <imstrmbl.hpp>

#if __IBMCPP__ >= 400
#pragma namemangling(compat)
#endif

#pragma enum(4)
#pragma pack(push,4)

/**
 * IAttributeState is a graphic state consisting of all the graphical state variables.
 * IAttributeState is a graphic state consisting of all the graphical state variables.
 * It is needed to draw a geometry, such as paint, transfer mode, pen geometry, matrix, and clip area.
 */

class IAttributeState : public IMSTREAMABLE
{
public:

	/** The EDrawOperation enum specifies the drawing operation. */
	enum EDrawOperation {
		kFrame,
		kFill,
		kFillAndFrame,
		kUnset
	};

   /**
	* Default constructor. Printing state is set to kPrintingUndefined.
	*/
	IAttributeState();

	/**
	 * Copy constructor
	 */
	IAttributeState(const IAttributeState&);

	virtual ~IAttributeState();

	/**
	 * Assignment operator.
	 */
	IAttributeState& operator=(const IAttributeState&);
	
	/**
	 * Returns the paint used for the surface fill operations.
	 */
	virtual const IPaint* fillPaint() const;
	
	/**
	 * Returns the color transfer mode to be used during a fill drawing operation.
	 */
	virtual const IColorTransferMode* fillTransferMode() const;
	
	/**
	 * Returns the paint used for the frame drawing operations.
	 */
	virtual const IPaint* framePaint() const;
	
	/**
	 * Returns the color transfer mode to be used during a frame drawing operation.
	 */
	virtual const IColorTransferMode* frameTransferMode() const;
	
	/**
	 * Returns the pen (such as IHairlinePen) to be used to draw the frame model of the graphic.
	 */
	virtual const IPen* framePen() const;
	
	/**
	 * Returns the cap (such as IFlushCap) to be used at the end of a line or geometry path.
	 */
	virtual const ICap* frameEndCap() const;
	
	/**
	 * Returns the joint to be used to smooth out the discontinuities which occur at the intersection of two line segments.
	 */
	virtual const IJoint* frameJoint() const;
	
	/**
	 * Returns the image transfer mode that determines how the colors of the image are modified when placed at the destination.
	 */
	virtual const IImageTransferMode* imageTransferMode() const;
	
	/**
	 * Returns the image sampling to be done when the graphic is either interpolated or decimated.
	 */
	virtual const IImageSamplingControl* imageSampling() const;
	
	/**
	 * Returns the drawing operation specified by the attribute state.
	 */
	virtual EDrawOperation drawingOperation() const = 0;

	/**
	 * Returns the time stamp stored by this object.
	 */
	virtual unsigned long timeStamp() const;
	
	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState& obj) const;

	/**
	 * Tests two IAttributeState objects for inequality.
	 */
	bool operator!=(const IAttributeState& obj) const;
	
//	StreamableDeclarationsMacro(IAttributeState);

protected:
    /** Streaming operations */
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);
	
	/**
	 * Sets the internal time stamp stored in this object.
	 */
	void setTimeStamp(unsigned long);
	
	/**
	 * Updates the time stamp value to a new value that is unique with respect to all other IAttributeState objects.
	 */
	void updateTimeStamp();

	/**
	 * Sets the time stamp to zero.
	 */
	void resetTimeStamp();

private:
friend class IPrintingFlagSnapshot;
	bool isPrinting() const;

friend class ILinkedBundleState;
friend class IPrintPreviewer;
friend class IInternalPrintChannel;
	enum EPrintingState {
		kNotPrinting,
		kIsPrinting,
		kPrintingUndefined
	};
	
	virtual EPrintingState printingFlag() const;
    void setPrintingFlag(EPrintingState printFlag);
	
private:
friend class IAttributeTimeStamp;
	static const unsigned long& staticAttTimeStamp();
	unsigned long fTimeStamp;
	EPrintingState fPrinting;
};

#if defined(_AIX) && IC_RTTI_ENABLED && __IBMCPP__ >= 360 && __IBMCPP__ < 400
/**
 * IBuggyBundle is a workaround for an AIX RTTI 96a compiler code generation bug.
 */

class IBuggyBundle : public IAttributeState
{
public:

	/**
	 * Default constructor. Creates an IBuggyBundle with black paint and color transfer mode as source-copy.
	 */
	IBuggyBundle();

	/**
	 * Copy constructor.
	 */
	IBuggyBundle(const IBuggyBundle&);

	/**
	 * Constructs an IBuggyBundle with a specified IBaseColor.
	 */
	IBuggyBundle(const IBaseColor& fillColor);

	/**
	 * Constructs an IBuggyBundle with specified color and color transfer mode.
	 */
	IBuggyBundle(
		const IBaseColor& fillColor,
		const IColorTransferMode& transferMode);

	virtual ~IBuggyBundle();

	/**
	 * Assignment operator.
	 */
	IBuggyBundle& operator=(const IBuggyBundle&);
	
	/**
	 * Sets the fill color for the object.
	 */
	void setFillColor(const IBaseColor& color);

	/**
	 * Returns the paint object being used for this object.
	 */
	virtual const IPaint* fillPaint() const;

	/**
	 * Sets the transfer mode that determines the fill color after deleting the existing one.
	 */
	void setFillTransferMode(const IColorTransferMode& transferMode);

	/**
	 * Returns a pointer to the transfer mode being used by this object.
	 */
	virtual const IColorTransferMode* fillTransferMode() const;

	/**
	 * Returns the EDrawOperation that determines whether to paint the fill, the frame, or both.
	 */
	virtual IAttributeState::EDrawOperation	drawingOperation() const;

	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState&) const;

	StreamableDeclarationsMacro(IBuggyBundle);

protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:
	IPaint fPaint;
	IColorTransferMode *fMode;
	IColorTransferMode fBaseMode;
	
	static IAttributeState::EDrawOperation fgDrawOperation;
};
#endif // defined(_AIX) && IC_RTTI_ENABLED && __IBMCPP__ >= 360 && __IBMCPP__ < 400

/**
 * IFillBundle contains imaging information for 2-D graphic objects of the solid-filled, no frame variety.
 */

class IFillBundle : public IAttributeState
{
public:

	/**
	 * Default constructor. Creates an IFillBundle with black paint and color transfer mode as source-copy.
	 */
	IFillBundle();

	/**
	 * Copy constructor.
	 */
	IFillBundle(const IFillBundle&);

	/**
	 * Constructs an IFillBundle with a specified IBaseColor.
	 */
	IFillBundle(const IBaseColor& fillColor);

	/**
	 * Constructs an IFillBundle with specified color and color transfer mode.
	 */
	IFillBundle(
		const IBaseColor& fillColor,
		const IColorTransferMode& transferMode);

	virtual ~IFillBundle();

	/**
	 * Assignment operator.
	 */
	IFillBundle& operator=(const IFillBundle&);
	
	/**
	 * Sets the fill color for the object.
	 */
	void setFillColor(const IBaseColor& color);

	/**
	 * Returns the paint object being used for this object.
	 */
	virtual const IPaint* fillPaint() const;

	/**
	 * Sets the transfer mode that determines the fill color after deleting the existing one.
	 */
	void setFillTransferMode(const IColorTransferMode& transferMode);

	/**
	 * Returns a pointer to the transfer mode being used by this object.
	 */
	virtual const IColorTransferMode* fillTransferMode() const;

	/**
	 * Returns the EDrawOperation that determines whether to paint the fill, the frame, or both.
	 */
	virtual IAttributeState::EDrawOperation	drawingOperation() const;

	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState&) const;

	StreamableDeclarationsMacro(IFillBundle);

protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:
	IPaint fPaint;
	IColorTransferMode *fMode;
	IColorTransferMode fBaseMode;
	
	static IAttributeState::EDrawOperation fgDrawOperation;
};

/**
 * IFrameBundle contains imaging information for 2D graphic objects of the non-filled, framed (with any size of frame) variety.
 */

class IFrameBundle : public IAttributeState
{

public:

	/**
	 * Default constructor. Creates an IFrameBundle with black paint, source-copy mode, and solid pen.
	 */
	IFrameBundle();

	/**
	 * Copy constructor.
	 */
	IFrameBundle(const IFrameBundle&);

	/**
	 * Constructs an IFrameBundle with specified frame color, pen width, and pen balance.
	 */
	IFrameBundle(const IBaseColor& frameColor,
		 GCoordinate penWidth = 1,
		 IPen::EPenBalance balance = IPen::kCenterFrame);

	/**
	 * Constructs an IFrameBundle with specified frame color, color transfer mode, pen width, and pen balance.
	 */
	IFrameBundle(const IBaseColor& frameColor,
		const IColorTransferMode& transferMode,
		GCoordinate penWidth = 1,
		IPen::EPenBalance balance = IPen::kCenterFrame);

	virtual ~IFrameBundle();

	/**
	 * Assignment operator.
	 */
	IFrameBundle& operator=(const IFrameBundle&);
	
	/**
	 * Sets the color for the frame.
	 */
	void setFrameColor(const IBaseColor& color);

	/**
	 * Returns the paint object being used for this frame.
	 */
	virtual const IPaint* framePaint() const;

	/**
	 * Returns a pointer to the color transfer mode being used by the frame of this graphic object.
	 */
	virtual const IColorTransferMode* frameTransferMode() const;
	
	/**
	 * Sets the transfer mode for the frame bundle after deleting the existing one.
	 */
	void setFrameTransferMode(const IColorTransferMode& transferMode);
	
	/**
	 * Sets the pen width for the frame.
	 */
	void setPenWidth(GCoordinate penWidth);
	
	/**
	 * Sets the pen balance for the frame.
	 */
	void setPenBalance(IPen::EPenBalance balance);

	/**
	 * Gets the IPen object that is used for the rendering of the frame.
	 */
	virtual const IPen* framePen() const;
	 	
	/**
	 * Returns the EDrawOperation that determines whether to paint the fill, the frame, or both.
	 */
	virtual IAttributeState::EDrawOperation drawingOperation() const;
	 	
	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState&) const;

	StreamableDeclarationsMacro(IFrameBundle);

protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:
	IPaint fPaint;

	IColorTransferMode* fMode;
	IColorTransferMode fBaseMode;
	IPen fPen;

	static IAttributeState::EDrawOperation fgDrawOperation;
};


/**
 * IFillAndFrameBundle contains imaging information for 2D graphic objects of the solid-filled, framed (with any size of frame) variety.
 */

class IFillAndFrameBundle : public IAttributeState
{
public:
	/**
	 * Default frame is black paint, source-copy color transfer mode, and solid pen.
	 * Default fill is black paint and source-copy color transfer mode.
	 */
	IFillAndFrameBundle();

	/**
	 * Copy constructor.
	 */
	IFillAndFrameBundle(const IFillAndFrameBundle&);

	/**
	 */
	IFillAndFrameBundle(const IBaseColor& fillColor,
			const IBaseColor& frameColor,
			GCoordinate penWidth = 1,
			IPen::EPenBalance balance = IPen::kCenterFrame);

	/**
	 */
	IFillAndFrameBundle(const IBaseColor& fillColor,
			const IColorTransferMode& fillTransferMode,
			const IBaseColor& frameColor,
			const IColorTransferMode& frameTransferMode,
			GCoordinate penWidth = 1,
			IPen::EPenBalance balance = IPen::kCenterFrame);

	virtual ~IFillAndFrameBundle();

	/**
	 * Assignment operator.
	 */
	IFillAndFrameBundle& operator=(const IFillAndFrameBundle&);
	
	/**
	 * Sets the fill color for the object.
	 */
	void setFillColor(const IBaseColor& color);
	
	/**
	 * Returns the paint object being used for the fill region of this graphic object.
	 */
	virtual const IPaint* fillPaint() const;
	
	/**
	 * Sets the transfer mode that determines the fill color after deleting the existing one.
	 */
	void setFillTransferMode(const IColorTransferMode& transferMode);
	
	/**
	 * Returns the paint object being used for the fill region of this graphic object.
	 */
	virtual const IColorTransferMode* fillTransferMode() const;
	
	/**
	 * Sets the color of the frame.
	 */
	void setFrameColor(const IBaseColor& color);
	
	/**
	 * Returns the paint object being used for this frame.
	 */
	virtual const IPaint* framePaint() const;
	
	/**
	 * Returns a pointer to the transfer mode being used by the frame of this graphic object.
	 */
	virtual const IColorTransferMode* frameTransferMode() const;
	
	/**
	 * Sets the transfer mode that determines the frame color after deleting the existing one.
	 */
	void setFrameTransferMode(const IColorTransferMode& transferMode);
	
	/**
	 * Sets the pen width for the frame.
	 */
	void setPenWidth(GCoordinate penWidth);
	
	/**
	 * Sets the pen balance for the frame.
	 */
	void setPenBalance(IPen::EPenBalance balance);
	
	/**
	 * Gets the IPen object that is used for the rendering of the frame.
	 */
	virtual const IPen* framePen() const;
	
	/**
	 * Returns the EDrawOperation that determines whether to paint the fill, the frame, or both.
	 */
	virtual IAttributeState::EDrawOperation	drawingOperation() const;
	
	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState&) const;

	StreamableDeclarationsMacro(IFillAndFrameBundle);

protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:

	IPaint fFillPaint;

	IColorTransferMode* fFillMode;
	IColorTransferMode fBaseFillMode;

	IPaint fFramePaint;
	IColorTransferMode* fFrameMode;
	IColorTransferMode fBaseFrameMode;

	IPen fPen;

	static IAttributeState::EDrawOperation fgDrawOperation;
};

/**
 * IGrafBundle is a collection of attributes that contain imaging information for the rendering device.
 */

class IGrafBundle : public IAttributeState
{
public :
	/**
	 * Default constructor with all attributes set to 0 (NIL).
	 */
	IGrafBundle();

	/**
	 * Creates an IGrafBundle using the given fill paint, frame paint, or both, as specified by the second parameter.
	 */
	IGrafBundle(const IPaint& fPaint,
		IAttributeState::EDrawOperation attribute = kFrame);

	/**
	 * Creates an IGrafBundle and adopts the given fill paint and frame paint.
	 */
	IGrafBundle(const IPaint& fillPaint,
		const IPaint& framePaint,
		IAttributeState::EDrawOperation attribute = kFillAndFrame);

	/**
	 * Creates an IGrafBundle of a specified color, which is to be used as an empty frame.
	 */
	IGrafBundle(const IBaseColor& color,
		IAttributeState::EDrawOperation attribute = kFrame);

	/**
	 * Creates an IGrafBundle of a specified color, which is to be used as a filled polygon with frame.
	 */
	IGrafBundle(const IBaseColor& fillColor,
		const IBaseColor& frameColor,
		IAttributeState::EDrawOperation attribute = kFillAndFrame);

	/**
	 * Copy constructor
	 */
	IGrafBundle(const IGrafBundle&);

	/**
	 * Creates an IGrafBundle, initialized with the data from the given attribute bundle.
	 */
	IGrafBundle(const IAttributeState&);

	virtual ~IGrafBundle();

	/**
	 * Returns the paint used for the fill (the area within the frame).
	 */
	virtual const IPaint* fillPaint() const;

	/**
	 * Sets the paint used for the fill (the area within the frame) after deleting the existing one.
	 */
	virtual void setFillPaint(const IPaint& paint);

	/**
	 * Delete the paint.
	 */
	virtual void removeFillPaint();

	/**
	 * Returns the color for the fill.
	 */
	virtual const IBaseColor* fillColor() const;

	/**
	 * Sets the color for the fill.
	 */
	virtual void setFillColor(const IBaseColor&);

	/**
	 * Returns the transfer mode for the fill.
	 */
	virtual const IColorTransferMode* fillTransferMode() const;

	/**
	 * Sets the transfer mode for the fill after deleting the existing one.
	 */
	virtual void setFillTransferMode(const IColorTransferMode&);

	/**
	 * Delete the transfer mode for the fill.
	 */
	virtual void removeFillTransferMode();
	
	/**
	 * Returns the paint used for the frame (the border of the graphic).
	 */
	virtual const IPaint* framePaint() const;

	/**
	 * Sets the frame paint after deleting the existing one.
	 */
	virtual void setFramePaint(const IPaint& paint);

	/**
	 * Delete the frame paint.
	 */
	virtual void removeFramePaint();
	
	/**
	 * Returns the frame color.
	 */
	virtual const IBaseColor* frameColor() const;

	/**
	 * Sets the frame color.
	 */
	virtual void setFrameColor(const IBaseColor&);

	/**
	 * Returns the transfer mode for the frame.
	 */
	virtual const IColorTransferMode* frameTransferMode() const;

	/**
	 * Sets the frame transfer mode for the frame after deleting the existing one.
	 */
	virtual void setFrameTransferMode(const IColorTransferMode&);

	/**
	 * Deletes the frame transfer mode.
	 */
	virtual void removeFrameTransferMode();
	
	/**
	 * Returns the pen used to draw the border of the graphic.
	 */
	virtual const IPen* framePen() const;

	/**
	 * Sets the pen used to draw the border of the graphic after deleting the existing one.
	 */
	virtual void setFramePen(const IPen& pen);

	/**
	 * Deletes the pen for the frame.
	 */
	virtual void removeFramePen();
	
	/**
	 * Returns a constant pointer to the ICap that determines the frame end cap style (such as flush or square).
	 */
	virtual const ICap*	frameEndCap() const;

	/**
	 * Sets the frame end cap style after deleting the existing one.
	 */
	virtual void setFrameEndCap(const ICap&);

	/**
	 * Delete the frame end cap style.
	 */
	virtual void removeFrameEndCap();
	
	/**
	 * Returns a constant pointer to the frame joint style (such as round or miter).
	 */
	virtual const IJoint* frameJoint() const;

	/**
	 * Sets the frame joint style after deleting the existing one.
	 */
	virtual void setFrameJoint(const IJoint&);

	/**
	 * Deletes the frame joint style.
	 */
	virtual void removeFrameJoint();
	
	/**
	 * Returns the transfer mode for an image after deleting the existing one.
	 */
	virtual const IImageTransferMode* imageTransferMode() const;

	/**
	 * Sets the transfer mode for an image after deleting the existing one.
	 */
	virtual void setImageTransferMode(const IImageTransferMode&);

	/**
	 * Delete the transfer mode for an image.
	 */
	virtual void removeImageTransferMode();
	
	/**
	 * Returns a constant pointer to the IImageSamplingControl that determines how the image used for the fill or frame is sampled.
	 */
	virtual const IImageSamplingControl* imageSampling() const;

	/**
	 * Sets the image sampling control that determines how the image used for the fill or frame is sampled after deleting the existing one.
	 */
	virtual void setImageSampling(const IImageSamplingControl&);

	/**
	 * Deletes the image sampling control that determines how the image used for the fill or frame is sampled.
	 */
	virtual void removeImageSampling();
	
	/**
	 * Returns the EDrawOperation that determines whether to paint the fill, the frame, or both.
	 */
	virtual IAttributeState::EDrawOperation drawingOperation() const;
	
	/**
	 * Specifies whether to paint the fill, the frame, or both.
	 */
	virtual void setDrawingOperation(IAttributeState::EDrawOperation);
	
	/**
	 * Assignment operator.
	 */
	IGrafBundle& operator=(const IGrafBundle& source);

	/**
	 * Tests two IAttributeState objects for equality.
	 */
	virtual bool operator==(const IAttributeState&) const;

	StreamableDeclarationsMacro(IGrafBundle);
	
protected :
	virtual void writeToStream(IDataStream& toWhere) const;
	virtual void readFromStream(IDataStream& toWhere);

private:
	IPaint* fFillPaint;
	IPaint fBaseFillPaint;

	IColorTransferMode* fFillMode;
	IColorTransferMode fBaseFillMode;

	IPaint* fFramePaint;
	IPaint fBaseFramePaint;

	IColorTransferMode* fFrameMode;
	IColorTransferMode fBaseFrameMode;

	IPen* fPen;
	IPen fBasePen;

	IJoint* fJoint;
	IJoint fBaseJoint;

	ICap* fEndCap;
	ICap fBaseEndCap;
	
	IImageTransferMode* fImageMode;
	IImageTransferMode fBaseImageMode;

	IImageSamplingControl* fSamplingControl;
	IImageSamplingControl fBaseSamplingControl;
	
	IAttributeState::EDrawOperation fDrawOperation;
};

#pragma pack(pop)
#pragma enum(pop)

#if __IBMCPP__ >= 400
#pragma namemangling()
#endif

#endif // _IBUNDLES_
