# Takes files or directories and detects image sequences. A sequence is
# several images in the time order where time interval between neighbouring
# images doesn't exceed the given maximum. Output sequences are sorted in
# exposure order.
# The usual take intervals on practice:
# - bracketing - 1 second
# - panorama - 5-6 seconds

import EXIF, calendar, getopt, glob, os, re, sys

def showUsage():
  print "Usage: " + sys.argv[0] + " [-s <seconds>] {files/directory names}\n" +\
        "    -s <seconds> Maximal count of seconds between shots in a sequence.\n" +\
        "        The default value is 2."
  sys.exit(0)

class Photo:
  def __init__(self, fileName, timestamp, exposure):
    self.fileName = fileName
    self.timestamp = timestamp
    self.exposure = exposure

  def __repr__(self):
    return self.fileName

def printSeq(photos, indFirst, indTerm):
  if indTerm < indFirst + 2:
    return # print only sequences with at least 2 items

  seq = photos[indFirst : indTerm]
  seq.sort(key=lambda a:a.exposure)

  for i in range(0, len(seq)):
    if i > 0:
      sys.stdout.write(" ")
    sys.stdout.write(seq[i].fileName)

  sys.stdout.write("\n")

def main():
  MAX_SEQ_INTERVAL = 2

  # parse options
  try:
    (optList, argFileNames) = getopt.getopt(sys.argv[1:], 's:')
  except:
    showUsage()

  opts={}
  for (optName, optVal) in optList:
    opts[optName] = optVal

  # time shift option
  if opts.has_key('-s'):
    MAX_SEQ_INTERVAL = int(opts['-s'])

  if not argFileNames:
    showUsage()

  fileNames = []
  for argFileName in argFileNames:
    if os.path.isdir(argFileName):
      dirFiles = glob.glob(os.path.join(argFileName, '*.*'))
      dirFiles.sort()
      fileNames = fileNames + dirFiles
    elif os.path.isfile(argFileName):
      fileNames.append(argFileName)
    else:
      sys.stderr.write("Filename %s is not found. Ignoring\n"%(argFileName))

  photos = []
  for fileName in fileNames:
    try:
      f = open(fileName, "rb")
      tags = EXIF.process_file(f)
      f.close()
    except:
      sys.stderr.write("Error parsing file %s: %s\n"%(fileName, `sys.exc_info()`))
      continue

    dateTagName = "EXIF DateTimeOriginal"
    if not tags.has_key(dateTagName):
      sys.stderr.write("%s doesn't contain creation date\n"%(fileName))
      continue

    dateTagVal = str(tags[dateTagName])
    dateMo = re.match("(\\d\\d\\d\\d):(\\d\\d):(\\d\\d) (\\d\\d):(\\d\\d):(\\d\\d)", dateTagVal)
    if not dateMo:
      sys.stderr.write("%s has not parsable date '%s'"%(fileName, dateStr))
      continue

    fileTimestamp = calendar.timegm((int(dateMo.group(1)), int(dateMo.group(2)), int(dateMo.group(3)),\
                                     int(dateMo.group(4)), int(dateMo.group(5)), int(dateMo.group(6))))

    # move extended (to all tags and bad values) version of this to EXIF.effectiveExposure()
    exposureTagName = "EXIF ExposureTime"
    if not tags.has_key(exposureTagName):
      sys.stderr.write("%s doesn't contain exposure time\n"%(fileName))
      continue

    fileExposure = EXIF.effectiveExposure(tags)

    photo = Photo(fileName, fileTimestamp, fileExposure)
    photos.append(photo)

  if not photos:
    return

  photos.sort(key=lambda a:a.timestamp) # ascending by timestamp

  indFirstInSeq = 0
  tsLastInSeq = photos[0].timestamp
  for i in range(1, len(photos)):
    photo = photos[i]
    if photo.timestamp <= tsLastInSeq + MAX_SEQ_INTERVAL:
      # continue sequence
      tsLastInSeq = photo.timestamp
      continue

    # sequence terminated
    printSeq(photos, indFirstInSeq, i) # print if it is not trivial
    indFirstInSeq = i # begin new sequence
    tsLastInSeq = photo.timestamp

  printSeq(photos, indFirstInSeq, len(photos)) # print last sequence

# ===================
main()
