import datetime
import os
import re
import sys
import glob
import getopt
import EXIF

OPTNAME_TIME_SHIFT="timeShift"
OPTNAME_PRINT_RENAME="printRename"
OPTNAME_SIMULATE_RENAME="simulateRename"
OPTNAME_GROUP_RENAME="groupRename"

manual="""
NAME
    exiftool - decode EXIF headers in JPEGs or TIFFs from digital cameras

SYNOPSIS
    exiftool [options] <files | directories> ...

DESCRIPTION
    Most digital cameras store additional information about each image
    directly within each file. This information is located in the so-called
    EXIF header and contains such things as exposure time, f-stop, "film"
    speed, etc.

    This tool decodes this information, and either prints it using a
    template, stores it in a file, or renames the file according to the
    date and time the image was taken.

OPTIONS
    -r              Rename files according to EXIF date.

    --%s=<seconds> On renaming adds given count of seconds to EXIF date
                    to form file name. A count may be negative.

    --%s Rename not only file with EXIF but also files with the same base
                    name and another extensions. Useful for RAWs or movies
                    which have an addiional JPEG file with the same name.

    --%s On renaming verbosely prints old and new file names.

    --%s On renaming performs all usual processing except actual rename.

    -w <filename wildcard> When dir is passed as argument take files only
                    matching this wildcard. *.jpg is probably the most
                    useful in practice.

    -f <file>       Specify template file.  Within the template, the value
                    of a tag can be printed by using the following syntax:

                            %%(<tagname>)s

                    e.g. %%(Image Model)s will report the camera model which
                    produced the file. For a list of the tagnames that can
                    be used, you can use the -v option.  If this option is
                    omitted, a default template is used.

                    Note that this can be used to write HTML files, with a
                    properly designed template file.

    -v              Verbose output. This reports all EXIF tags that are
                    found in the file.

    -i <ext>        Write information from each image file to an individual
                    file with an extension of <ext>, e.g. "-i txt"
                    specifies that information from "foo.jpg" is written to
                    "foo.txt".  If the special name "stdout" is used, the
                    output is written to stdout.

    -t              Extract thumbnail, if any.  If file is named "foo.tif"
                    and the thumbnail is in JPEG format, the thumbnail will
                    be named "foo_thumb.jpg"

AUTHOR
    Gene Cash <gcash@cfl.rr.com>
    Slavik Gnatenko <moveton@gmail.com>
"""%(OPTNAME_TIME_SHIFT, OPTNAME_GROUP_RENAME, OPTNAME_PRINT_RENAME, OPTNAME_SIMULATE_RENAME)

# default template
template="""
EXIF information in "%(Filename)s"
          Camera: %(Image Make)s %(Image Model)s
            Date: %(EXIF DateTimeOriginal)s
      Image size: %(EXIF ExifImageWidth)sx%(EXIF ExifImageLength)s pixels
   Exposure Time: %(EXIF ExposureTime)s
        Aperture: f/%(EXIF FNumber)s
Exposure Program: %(EXIF ExposureProgram)s
   Exposure Bias: %(EXIF ExposureBiasValue)s
   Metering Mode: %(EXIF MeteringMode)s
           Flash: %(EXIF Flash)s
    Focal Length: %(EXIF FocalLength)s
"""

# handle missing tags gracefully
class PrintMap:
    def __init__(self, map):
    	self.map=map
        
    def __getitem__(self, key):
    	return self.map.get(key, '???')

# convert date to string that's a legal filename
# "2001:03:31 12:27:36" becomes "010331_122736"
def date2file(d):
    dateStr=str(d)
    if not timeShiftSeconds:
        dateStr=dateStr.replace(' ', '_')
        dateStr=dateStr.replace(':', '')
        dateStr=dateStr[2:len(dateStr)]
    else:
        dateMo = re.match("(\\d\\d\\d\\d):(\\d\\d):(\\d\\d) (\\d\\d):(\\d\\d):(\\d\\d)", dateStr)
        if not dateMo:
            print "%s is not parsable date"%(dateStr)
            dateStr=None
        else:
            dateDT = datetime.datetime(int(dateMo.group(1)), int(dateMo.group(2)), int(dateMo.group(3)), int(dateMo.group(4)), int(dateMo.group(5)), int(dateMo.group(6)))
            dateDT = dateDT + datetime.timedelta(0, timeShiftSeconds)
            dateStr = dateDT.strftime("%y%m%d_%H%M%S")

#    print "'%s' converted to '%s'"%(str(d), dateStr)
    return dateStr

# Constructs full file name from directory name, base name, extension and
# an additional index. If the index is zero it is not appended to the name.
def constructPathName(dirName, baseName, nameExt, nameInd):
    if nameInd:
        nameIndStr = "_%03d"%(nameInd)
    else:
        nameIndStr = ""

    return os.path.join(dirName, baseName)+nameIndStr+nameExt


# start of main code
if len(sys.argv) == 1:
    print manual
    sys.exit()

# parse options
try:
    optlist, args=getopt.getopt(sys.argv[1:], 'f:i:rs:tv',
            [OPTNAME_TIME_SHIFT+'=', OPTNAME_GROUP_RENAME, OPTNAME_PRINT_RENAME, OPTNAME_SIMULATE_RENAME])
except:
    print manual
    sys.exit()
     
opts={}
for i in optlist:
    opts[i[0]]=i[1]

# list parsed tags
verbose=0
if opts.has_key('-v'):
    verbose=1

# rename file according to date
rename=0
timeShiftSeconds=0
if opts.has_key('-r'):
    rename=1
    optTimeShiftSeconds=opts.get('--'+OPTNAME_TIME_SHIFT)
    if optTimeShiftSeconds != None:
        timeShiftSeconds=int(optTimeShiftSeconds)

optGroupRename=opts.has_key('--'+OPTNAME_GROUP_RENAME)
optPrintRename=opts.has_key('--'+OPTNAME_PRINT_RENAME)
optSimulateRename=opts.has_key('--'+OPTNAME_SIMULATE_RENAME)

# filename wildcard. By default - all filenames with suffix
optFilenameWildcard=opts.get('-w', '*.*')

# extract thumbnail
thumb=0
if opts.has_key('-t'):
    thumb=1

# template file
if opts.has_key('-f'):
    template=open(opts['-f']).read()

# extension
optInfoExt=opts.get("-i")

# process all files/directories given
files=[]
for i in args:
    if os.path.isdir(i):
        dirFiles = glob.glob(os.path.join(i, optFilenameWildcard))
        dirFiles.sort()
        files=files+dirFiles
    elif os.path.isfile(i):
        files.append(i)

handledPathNames = set()
for path_name in files:
    if path_name in handledPathNames:
        continue

    (dir_name, file_name) = os.path.split(path_name)
    (photo_name, photo_ext) = os.path.splitext(file_name)

    try:
        f=open(path_name, 'rb')
    except IOError, exc:
        print "Failed to open %s: %s. Skipping"%(path_name, str(exc))
        continue

    tags=EXIF.process_file(f)
    f.close()
    
    # rename files according to date
    if rename and tags.has_key('EXIF DateTimeOriginal'):
        new_name=date2file(tags['EXIF DateTimeOriginal'])
        if not new_name:
            print "Failed to generate new file name for %s"%(new_name)
        elif (new_name == photo_name):
            print "Rename of %s is skipped because new name is the same"%(path_name)
        else:
            if optGroupRename and photo_ext: # only name with extension can be group anchor
                # take all names which differ only by extension. Calling glob()
                # also handles case sensitivity platform issues
                fullNamesOld = glob.glob(os.path.join(dir_name, photo_name + ".*"))
            else:
                # no actual group handling. Create trivial group with only original name
                fullNamesOld = [path_name]

            exts = []
            for fullNameOld in fullNamesOld:
                (tmpName, tmpExt) = os.path.splitext(fullNameOld)
                exts.append(tmpExt)

            # search for free index
            nameIndex = -1
            for nameInd in range(1000): # assume this is enough for 1 second name clash
                for ext in exts:
                    fullNameNew = constructPathName(dir_name, new_name, ext, nameInd)
                    if os.access(fullNameNew, os.F_OK):
                        break
                else:
                    # found free index
                    nameIndex = nameInd
                    break

            if nameIndex < 0:
                print "Rename of %s is skipped because all targets like %s_N already exist"%(path_name, new_name)
            else:
                # OK index is found. Perform rename
                photoRenamed = False
                for ext in exts:
                    fullNameOld = os.path.join(dir_name, photo_name) + ext
                    fullNameNew = constructPathName(dir_name, new_name, ext, nameIndex)

                    if optPrintRename:
                        print "Renaming %s to %s"%(fullNameOld, fullNameNew)

                    if not optSimulateRename:
                        try:
                            os.rename(fullNameOld, fullNameNew)
                        except OSError, exc:
                            print "Rename of %s to %s is failed: %s"%(fullNameOld, fullNameNew, str(exc))
                        else:
                            if ext == photo_ext:
                                # path_name has been renamed
                                photoRenamed = True
                            else:
                                # not path_name has been renamed. Possibly this
                                # name will occur in the files list later. To
                                # avoid IOError on its open we save it to the
                                # ignore list
                                handledPathNames.add(fullNameOld)

                # successfull rename. Switch names variables to the new name. It
                # is necessary for thumbs processing. Switch can be done only
                # here because photo_name is used in other renames in the group
                if photoRenamed:
                    photo_name=new_name
                    file_name=photo_name+photo_ext

    # write extracted thumbnail
    if thumb:
        if tags.has_key('JPEGThumbnail'):
            thumb_name=os.path.join(dir_name, photo_name)+'_thumb.jpg'
            open(thumb_name, 'wb').write(tags['JPEGThumbnail'])
        if tags.has_key('TIFFThumbnail'):
            thumb_name=os.path.join(dir_name, photo_name)+'_thumb.tif'
            open(thumb_name, 'wb').write(tags['TIFFThumbnail'])

    tags['Filename']=file_name
    if verbose:
        # print raw tags, sorted by name
        k=tags.keys()
        k.sort()
        # determine length of longest tag name
        cols=max(map(len, k))
        # print file name first
        print '%*s: %s' % (cols, 'Filename', tags['Filename'])

        # print all tags
        for i in k:
            if i not in ('JPEGThumbnail', 'TIFFThumbnail', 'Filename'):
                print '%*s: %s' % (cols, i, tags[i])

        # thumbnail information
        print '%*s:' % (cols, 'Includes JPEG thumbnail'),
        if tags.has_key('JPEGThumbnail'):
            print 'Yes'
        else:
            print 'No'
        print '%*s:' % (cols, 'Includes TIFF thumbnail'),
        if tags.has_key('TIFFThumbnail'):
            print 'Yes'
        else:
            print 'No'
        print

    # print info using template
    if optInfoExt:
        if optInfoExt == 'stdout':
            print template % PrintMap(tags)
        else:
            out_name=os.path.join(dir_name, photo_name)+'.'+optInfoExt
            open(out_name, 'wb').write(template % PrintMap(tags))
